/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.graph;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import xyz.cofe.collection.Convertor;
import xyz.cofe.collection.Iterators;
import xyz.cofe.collection.Predicate;


/**
 * Поиск путей в графе. <p>
 * Производит обход графа формируя возможные пути.
 * Обход производится по крайчащим путям.
 * Конечная вершина поиска определяется пользователем данного класса. <p>
 * За один вызов next() выдает один возможный путь из указанной точки.
 * @author gocha
 * @param <N> Тип вершины
 * @param <E> Тип ребра
 */
public class PathFinder<N,E> implements Iterator<Path<N,E>> 
{
//    /**
//     * Полчение веса пути
//     * @param path Путь
//     * @param getWeight Получение веса ребра
//     * @param initWeight Начальный вес (нулевой вес)
//     * @param weightSummator Функция - Сумматор веса
//     * @return Вес пути
//     */
//    protected static <W> W getWeightOf( Path path, 
//        Convertor getWeight, 
//        W initWeight,
//        Func2<W,W,W> weightSummator 
//    ){
//        W w = initWeight;
//        for( Object oe : path ){
//            W we = (W)getWeight.convert(oe);
//            w = weightSummator.apply(w, we);
//        }
//        return w;
//    }
//    
//    protected class PathWeightComparator<W extends Comparable<W>>
//    implements Comparator<Path<N,E>>
//    {
//        public W initWeight;
//        public Func2<W,W,W> weightSummator;
//        public Convertor<Edge<N,E>, W> getWeight;
//        
//        @Override
//        public int compare(Path p1, Path p2) {
//            W w1 = getWeightOf(p1, getWeight, initWeight, weightSummator);
//            W w2 = getWeightOf(p2, getWeight, initWeight, weightSummator);
//            return w1.compareTo(w2);
//        }
//    }
//
//    /**
//     * Конструктор
//     * @param graph Одно направленный граф
//     * @param start Начальная вершина пути
//     * @param direction Направление движения
//     * @param initWeight Начальный вес (нулевой вес)
//     * @param getWeight Получение веса ребра, вес должен быть равным или больше initWeight
//     * @param weightSummator Суммирование веса
//     */
//    public static <N,E,W extends Comparable> PathFinder<N,E> create(
//        SingleDirectedGraph<N,E> graph,
//        N start,
//        Path.Direction direction,
//        W initWeight,
//        Convertor<Edge<N,E>,W> getWeight,
//        Func2<W,W,W> weightSummator
//    ){
//        PathFinder pf = new PathFinder();
//        pf.
//        
//        return null;
//    }
//    
//    /**
//     * Конструктор
//     */
//    protected PathFinder(){
//    }
    
    /**
     * Конструктор
     * @param graph Одно направленный граф
     * @param start Начальная вершина пути
     * @param direction Направление движения
     * @param getWeight Получение веса ребра, вес должен быть положительным или равен нулю
     */
    public PathFinder(
        SingleDirectedGraph<N,E> graph,
        N start,
        Path.Direction direction,
        Convertor<Edge<N,E>,Double> getWeight
    ){
        if( getWeight==null )throw new IllegalArgumentException( "getWeight==null" );
        if( start==null )throw new IllegalArgumentException( "start==null" );
        if( direction==null )throw new IllegalArgumentException( "direction==null" );
        if( graph==null )throw new IllegalArgumentException( "graph==null" );
        
        this.graph = graph;
        this.direction = direction;
        this.paths = createPathsList();
        this.comparator = createComparatorFrom(getWeight);
        this.validator = createValidtor();

        Iterable<Edge<N,E>> next = getNextEdges(start);
        for( Edge<N,E> e : next ){
            Path<N,E> bp = createPath(direction);
            bp.add(e);
            paths.add(bp);
        }
        Collections.sort(paths, comparator);
        pathItr = paths.iterator();
    }
    
    /**
     * Полчение веса пути
     * @param path Путь
     * @param getWeight Получение веса ребра
     * @return Вес пути
     */
    protected double getIntWeightOf( Path<N,E> path, Convertor<Edge<N,E>,Double> getWeight ){
        double w = 0;
        for( Edge<N,E> e : path ){
            double we = getWeight.convert(e);
            w += we;
        }
        return w;
    }
    
    /**
     * Создание Comparator для пути
     * @param getWeight Получение веса ребра
     * @return Comparator
     */
    protected Comparator<Path<N,E>> createComparatorFrom( Convertor<Edge<N,E>,Double> getWeight ){
        final Convertor<Edge<N,E>,Double> fgetWeight = getWeight;
        return new Comparator<Path<N, E>>() {
            @Override
            public int compare(Path<N, E> p1, Path<N, E> p2) {
                double w1 = getIntWeightOf(p1, fgetWeight);
                double w2 = getIntWeightOf(p1, fgetWeight);
                return w1 == w2 ? 0 : ( w1 < w2 ? -1 : 1 );
            }
        };
    }
    
    /**
     * Конструктор
     * @param graph Одно направленный граф
     * @param start Начальная вершина пути
     * @param direction Направление движения
     * @param comparator Сравнение длины путей
     */
    public PathFinder(
            SingleDirectedGraph<N,E> graph,
            N start,
            Path.Direction direction,
            Comparator<Path<N,E>> comparator
            ){
        if (graph== null) {            
            throw new IllegalArgumentException("graph==null");
        }
        if (start== null) {            
            throw new IllegalArgumentException("start==null");
        }
        if (direction== null) {            
            throw new IllegalArgumentException("direction==null");
        }
        if (comparator== null) {            
            throw new IllegalArgumentException("comparator==null");
        }

        this.graph = graph;
        this.direction = direction;
        this.paths = createPathsList();
        this.comparator = comparator;
        this.validator = createValidtor();

        Iterable<Edge<N,E>> next = getNextEdges(start);
        for( Edge<N,E> e : next ){
            Path<N,E> bp = createPath(direction);
            bp.add(e);
            paths.add(bp);
        }
        Collections.sort(paths, comparator);
        pathItr = paths.iterator();
    }

    /**
     * Создает список путей
     * @return Список путей
     */
    protected List<Path<N,E>> createPathsList(){ return new ArrayList<Path<N, E>>(); }
    
    /**
     * Создаает путь
     * @param d Направление
     * @return Путь
     */
    protected Path<N,E> createPath(Path.Direction d){
        BasicPath<N,E> p = new BasicPath<N, E>();
        p.setDirection(d);
        return p;
    }

    /**
     * Добавляет ребро в конец пути
     * @param path Путь
     * @param e Ребро
     * @return Новый путь
     */
    protected Path<N,E> append(Path<N,E> path,Edge<N,E> e){
        BasicPath<N,E> npath = new BasicPath<N, E>();
        npath.addAll(path);
        npath.add(e);
        return npath;
    }

    /**
     * Граф в котором производится поиск
     */
    protected SingleDirectedGraph<N,E> graph = null;
    
    /**
     * Направление движения
     */
    protected Path.Direction direction = null;
    
    /**
     * Список путей используемых в поиске.
     * Используются их конечные вершины.
     */
    protected List<Path<N,E>> paths = null;
    protected Comparator<Path<N,E>> comparator = null;
    protected Iterator<Path<N,E>> pathItr = null;
    protected Predicate<Edge<N,E>> validator = null;

    /**
     * Создает предикат проверки циклов в пути.
     * @return true - Проверяемое ребро AB <b>не</b> существует в пути; <p>
     * false - Проверяемое ребро AB <b>существует</b> в пути;
     */
    protected Predicate<Edge<N,E>> createValidtor(){
        Predicate<Edge<N,E>> p = new Predicate<Edge<N,E>>() {
            @Override
            public boolean validate(Edge<N,E> value) {
                if( paths==null )return false;
                for( Path<N,E> p : paths ){
                    if( p.contains(value.getNodeA(), value.getNodeB()) )
                        return false;
                }
                return true;
            }
        };
        return p;
    }
    
    /**
     * Извлекает исходящие ребра/дуги из вершины n в соот. движению.
     * @param n Вершина
     * @return Ребра/дуги направления движения.
     */
    protected Iterable<Edge<N,E>> getNextEdges(N n)
    {
        Iterable<Edge<N,E>> itr = direction.equals(Path.Direction.AB) ? graph.edgesOfNodeA(n) : graph.edgesOfNodeB(n);
        if( validator!=null ){
            itr = Iterators.predicate(itr, validator);
        }
        return itr;
    }
    
//    /**
//     * Проверка возможности добавления ребра в путь, для исключения циклов
//     * @param path путь
//     * @param e ребро
//     * @return true - добавлять можно
//     */
//    protected boolean allowAppend( Path<N,E> path, Edge<N,E> e ){
//        if( path==null )return false;
//        if( e==null )return false;
//        
//        if( validator!=null ){
//            if( !validator.validate(e) )return false;
//            return true;
//        }
//        
//        N na = e.getNodeA();
//        if( na==null )return false;
//        
//        for( Edge<N,E> pe : path ){
//            N ea = pe.getNodeA();
//            if( ea==null )return false;
//            
//            if( ea==na )return false;
//            if( ea.equals(na) )return false;
//        }
//        
//        return false;
//    }

    /**
     * Производит поиск путей. <p>
     * 1. Из списка путей извлекает первый путь и удаляет его из списка; <p>
     * 2. Ищет возможные дальнейшие пути из последней вершины, добавляя найденые в конец списка; <p>
     * 3. Сортирует список путей по возрастанию.
     */
    protected void searchAgain(){
        if( paths.size()<1 ){
            pathItr = null;
            return;
        }

        Path<N,E> smallPath = paths.get(0);
        Iterable<Edge<N,E>> next = getNextEdges(smallPath.getLastNode());
        for( Edge<N,E> e : next ){
//            if( allowAppend(smallPath, e) ){
                paths.add( append(smallPath, e) );
//            }
        }
        paths.remove(0);
        Collections.sort(paths, comparator);
        pathItr = paths.iterator();
    }

    /* (non-Javadoc)
     * @see java.util.Iterator
     */
    @Override
    public boolean hasNext() {
        if( pathItr==null )return false;
        boolean res = pathItr.hasNext();
        if( res==false )searchAgain();
        return res;
//        throw new UnsupportedOperationException("Not supported yet.");
    }

    /* (non-Javadoc)
     * @see java.util.Iterator
     */
    @Override
    public Path<N, E> next() {
        if( pathItr==null )return null;
        
        Path<N,E> p = pathItr.next();
        if( !pathItr.hasNext() )searchAgain();

        return p;
    }

    /* (non-Javadoc)
     * @see java.util.Iterator
     */
    @Override
    public void remove() {
//        throw new UnsupportedOperationException("Not supported yet.");
    }
}
