/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.iterators;

import java.util.ArrayList;
import xyz.cofe.collection.Convertor;
import xyz.cofe.collection.Iterators;

/**
 * Реализация интерфейса информации о обходе дерева - текущем узле/листе,
 * с возможностью изменения.
 * @author gocha
 * @param <T> Тип узла древа
 */
public class MutableTreeWalk<T> implements TreeWalk<T>
{
    /**
     * Конструктор жочернего узла
     * @param parentTreeWalk Родительский узел
     * @param currentNode Текущий узел
     * @param currentLevel Текущий уровень
     * @param stepLevel Шаг между уровнями
     */
    public MutableTreeWalk(TreeWalk<T> parentTreeWalk,T currentNode,int currentLevel,int stepLevel)
    {
        this.parent = parentTreeWalk;
        this.currentNode = currentNode;
        this.currentLevel = currentLevel;
        this.stepLevel = stepLevel;

        if( parentTreeWalk!=null )
        {
            this.parentNode = parentTreeWalk.currentNode();
        }

        // Составление пути
        ArrayList<TreeWalk<T>> _path = new ArrayList<TreeWalk<T>>();
        TreeWalk<T> n = this;
        _path.add(n);
        while( n.parent()!=null )
        {
            TreeWalk<T> prnt = n.parent();
            _path.add(0,prnt);
            n = prnt;
        }

        if( this.parent!=null && _path!=null && _path.size()>0 )
        {
            this.startNode = _path.get(0).currentNode();
            this.startLevel = _path.get(0).startLevel();
        }

        this.walkPath = _path;
    }

    /**
     * Конструктор начального узла (корня) дерева обхода
     * @param startNode Узел (корень)
     * @param startLevel начальный уровень
     * @param stepLevel Шаг между уровнями
     */
    public MutableTreeWalk(T startNode, int startLevel, int stepLevel)
    {
        this.currentNode = startNode;
        this.parentNode = null;
        this.startNode = startNode;
        this.currentLevel = startLevel;
        this.startLevel = startLevel;
        this.stepLevel = stepLevel;

        walkPath = Iterators.<TreeWalk<T>>single(this);
    }

    /**
     * Путь от корня, до текущего узла
     */
    protected Iterable<TreeWalk<T>> walkPath = null;

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    @Override
    public Iterable<TreeWalk<T>> walkPath()
    {
        return walkPath;
    }

    private Convertor<TreeWalk<T>,T> convertor2node = new Convertor<TreeWalk<T>,T>()
    {
        @Override
        public T convert(TreeWalk<T> from)
        {
            return from.currentNode();
        }
    };

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    @Override
    public Iterable<T> nodePath()
    {
        return Iterators.<TreeWalk<T>,T>convert(walkPath, convertor2node);
    }

    /** Родительский узел (или null) */
    protected TreeWalk<T> parent = null;

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    @Override
    public TreeWalk<T> parent()
    {
        return parent;
    }

    /**
     * Текущий узел/лист древа
     */
    protected T currentNode = null;

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    @Override public T currentNode() { return currentNode; }

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    public void currentNode(T newCurrentNode ){ currentNode = newCurrentNode; }

    /**
     * Текущий уровень
     */
    protected int currentLevel = 0;

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    @Override public int currentLevel() { return currentLevel; }

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    public void currentLevel(int newCurrentLevel ) { currentLevel = newCurrentLevel; }

    /**
     * Родительский узел (или null)
     */
    protected T parentNode = null;

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    @Override public T parentNode() { return parentNode; }

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    public void parentNode(T newParentNode) { parentNode = newParentNode; }

    /**
     * Корневой узел (или null)
     */
    protected T startNode = null;

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    @Override public T startNode() { return startNode; }

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    public void startNode(T newStartNode) { startNode = newStartNode; }

    /**
     * Начальный уровень
     */
    protected int startLevel = 0;

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    @Override public int startLevel() { return startLevel; }

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    public void startLevel(int newStartLevel) { startLevel = newStartLevel; }

    /**
     * Шаг между уровнями
     */
    protected int stepLevel = 1;

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    @Override public int stepLevel() { return stepLevel; }

    /* (non-Javadoc)
     * @see org.gocha.collection.iterators.TreeWalk */
    public void stepLevel(int newStepLevel) { stepLevel = newStepLevel; }

    protected boolean firstChild = false;
    protected boolean lastChild = false;
    protected int childIndex = -1;

    /**
     * Возвращает индекс узла среди соседних элементов по порядку обхода. <p>
     * Пример:
     * <table border="1" summary="Распределение индексов">
     * <tr>
     * <td colspan="4">
     * 0
     * </td>
     * </tr>
     * <tr>
     * <td colspan="3">0</td><td>1</td>
     * </tr><tr>
     * <td>0</td><td>1</td><td>2</td><td>0</td>
     * </tr>
     * </table>
     * @return Индекси или -1
     */
    public int getChildIndex()
    {
        return childIndex;
    }

    /**
     * Устанавливает индекс узла среди соседних элементов по порядку обхода.
     * @param childIndex Индекси или -1
     */
    public void setChildIndex(int childIndex)
    {
        this.childIndex = childIndex;
    }

    /**
     * Возвращает признак что данный узел является  первым дочерним узлом среди соседниъ элементов.
     * <table border="1" summary="Схема First/Last">
     * <tr>
     * <td colspan="4">0</td>
     * </tr>
     * <tr>
     * <td colspan="2">1 F</td><td colspan="2">4 L</td>
     * </tr>
     * <tr>
     * <td>2 F</td><td>3 L</td>
     * <td>5 F</td><td>6 L</td>
     * </tr>
     * </table>
     * @return true - является первым; не является
     */
    public boolean isFirstChild()
    {
        return firstChild;
    }

    /**
     * Устанавливает признак что данный узел является  первым дочерним узлом среди соседних элементов.
     * @param firstChild true - является первым; не является
     * @see #isFirstChild()
     */
    public void setFirstChild(boolean firstChild)
    {
        this.firstChild = firstChild;
    }

    /**
     * Возвращает признак что данный узел является  последним дочерним узлом среди соседних элементов.
     * @return true - является последним; false - не является
     * @see #isFirstChild()
     */
    public boolean isLastChild()
    {
        return lastChild;
    }

    /**
     * Устанавливает признак что данный узел является  последним дочерним узлом среди соседних элементов.
     * @param lastChild  true - является последним; false - не является
     * @see #isFirstChild()
     */
    public void setLastChild(boolean lastChild)
    {
        this.lastChild = lastChild;
    }
}
