/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.iterators;

import java.util.ArrayDeque;
import java.util.Iterator;
import xyz.cofe.collection.Iterators;
import xyz.cofe.collection.NodesExtracter;
import xyz.cofe.collection.Predicate;

/**
 * Обход дерева
 * @author gocha
 */
public class TreeWalkItreator<T> implements Iterator<TreeWalk<T>>
{
    private ArrayDeque<MutableTreeWalk<T>> nodesQueue = null;
    private NodesExtracter<T,T> extracter = null;
    private TreeWalkType walkType = TreeWalkType.ByBranchForward;
    private Predicate<T> nodePredicate = null;
    private Predicate<TreeWalk<T>> walkPredicate = null;
//    private Visitor<T> visiter = null;

//    public

    /**
     * Создает последовательность
     * @param <T> Тип объекта в последовательности
     * @param startNode Начальный узел
     * @param extracter Интерфес доступа к дочерним узлам
     * @return Последовательность
     */
    public static <T> Iterable<TreeWalk<T>> createIterable(T startNode,NodesExtracter<T,T> extracter)
    {
        final T fstartNode = startNode;
        final NodesExtracter<T,T> fextracter = extracter;
        return new Iterable<TreeWalk<T>>() {
            @Override
            public Iterator<TreeWalk<T>> iterator()
            {
                return new TreeWalkItreator(fstartNode, fextracter);
            }
        };
    }

    /**
     * Создает последовательность
     * @param <T> Тип объекта в последовательности
     * @param startNode Начальный узел
     * @param extracter Интерфес доступа к дочерним узлам
     * @param walkType Тип обхода
     * @return Последовательность
     */
    public static <T> Iterable<TreeWalk<T>> createIterable(
            T startNode,
            NodesExtracter<T,T> extracter,
            TreeWalkType walkType)
    {
        final T fstartNode = startNode;
        final NodesExtracter<T,T> fextracter = extracter;
        final TreeWalkType fWalkType = walkType;

        return new Iterable<TreeWalk<T>>() {
            @Override
            public Iterator<TreeWalk<T>> iterator()
            {
                return new TreeWalkItreator(
                        fstartNode, fextracter, 0, 1,
                        fWalkType, null, null
                        );
            }
        };
    }

    /**
     * Создает последовательность
     * @param <T> Тип объекта в последовательности
     * @param startNode Начальный узел
     * @param extracter Интерфес доступа к дочерним узлам
     * @param walkPredicate Предикат обхода узела
     * @return Последовательность
     */
    public static <T> Iterable<TreeWalk<T>> createIterable(
            T startNode,
            NodesExtracter<T,T> extracter,
            Predicate<TreeWalk<T>> walkPredicate)
    {
        final T fstartNode = startNode;
        final NodesExtracter<T,T> fextracter = extracter;
        final Predicate<TreeWalk<T>> fwalkPredicate = walkPredicate;
        return new Iterable<TreeWalk<T>>() {
            @Override
            public Iterator<TreeWalk<T>> iterator()
            {
                return new TreeWalkItreator(
                        fstartNode, fextracter, 0, 1,
                        TreeWalkType.ByBranchForward, null, fwalkPredicate);
            }
        };
    }

    /**
     * Создает последовательность
     * @param <T> Тип объекта в последовательности
     * @param startNode Начальный узел
     * @param extracter Интерфес доступа к дочерним узлам
     * @param walkPredicate Предикат обхода узела
     * @param walkType Тип обхода
     * @return Последовательность
     */
    public static <T> Iterable<TreeWalk<T>> createIterable(
            T startNode,
            NodesExtracter<T,T> extracter,
            Predicate<TreeWalk<T>> walkPredicate,
            TreeWalkType walkType)
    {
        final T fstartNode = startNode;
        final NodesExtracter<T,T> fextracter = extracter;
        final Predicate<TreeWalk<T>> fwalkPredicate = walkPredicate;
        final TreeWalkType fWalkType = walkType;

        return new Iterable<TreeWalk<T>>() {
            @Override
            public Iterator<TreeWalk<T>> iterator()
            {
                return new TreeWalkItreator(
                        fstartNode, fextracter, 0, 1,
                        fWalkType, null, fwalkPredicate);
            }
        };
    }

    /**
     * Конструктор
     * @param startNode Начальный узел обхода дерева
     * @param extracter Интерфес доступа к дочерним узлам
    */
    public TreeWalkItreator(T startNode,NodesExtracter<T,T> extracter)
    {
        this( startNode, extracter, 0, 1, TreeWalkType.ByBranchForward, null, null );
    }

    /**
     * Конструктор
     * @param startNode Начальный узел обхода дерева
     * @param extracter Интерфес доступа к дочерним узлам
     * @param startLevel Начальный уровень
     * @param levelStep Шаг уровня
     * @param walkType Тип обхода (Может быть null)
     * @param nodePredicate Предикат обхода узела (Может быть null)
     * @param walkPredicate Предикат обхода узела (Может быть null, если не сработал nodePredicate, то отрабатывает этот предикат)
     */
    public TreeWalkItreator(
            T startNode,
            NodesExtracter<T,T> extracter,
            int startLevel,
            int levelStep,
            TreeWalkType walkType,
            Predicate<T> nodePredicate,
            Predicate<TreeWalk<T>> walkPredicate
            )
    {
        if (startNode == null) {
            throw new IllegalArgumentException("startNode == null");
        }
        if (extracter == null) {
            throw new IllegalArgumentException("extracter == null");
        }

        this.extracter = extracter;

        if( walkType==null )walkType = TreeWalkType.ByBranchForward;
        this.walkType = walkType;

        nodesQueue = new ArrayDeque<MutableTreeWalk<T>>();

        this.nodePredicate = nodePredicate;
        this.walkPredicate = walkPredicate;

        //======================
        boolean add = true;
        if( nodePredicate!=null )
            if( !nodePredicate.validate(startNode) )
                add = false;

        MutableTreeWalk<T> mtw = new MutableTreeWalk<T>(startNode, startLevel, levelStep);

        if( walkPredicate!=null )
            if( !walkPredicate.validate(mtw) )
                add = false;

        if( add )nodesQueue.add(mtw);
    }

    @Override
    public boolean hasNext() {
        return !nodesQueue.isEmpty();
    }

    @Override
    public TreeWalk<T> next() 
    {
        switch( walkType )
        {
            case ByLevel: return topDown();
            case ByBranchBackward: return deepParentChildBack();
            case ByBranchForward: return deepParentChildForward();
        }
        return deepParentChildForward();
    }

    private TreeWalk<T> deepParentChildForward()
    {
        if( nodesQueue.isEmpty() )return null;

        MutableTreeWalk<T> result = nodesQueue.poll();
        T currentNode = result.currentNode();
//        T startNode = result.startNode();
        int step = result.stepLevel();
//        int startL = result.startLevel();

        Iterable<T> nodes = extracter.extract(currentNode);
        if( nodes!=null )nodes = Iterators.<T>reverse(nodes);
        if( nodes!=null )
        {
            int idx = -1;
            MutableTreeWalk<T> mtw = null;

            for( T n : nodes )
            {
                if( nodePredicate!=null )
                    if( !nodePredicate.validate(n) )
                        continue;

                mtw =
                new MutableTreeWalk<T>(
                        result,
                        n,
                        result.currentLevel()+step,
                        step
                        );

                if( walkPredicate!=null )
                    if( !walkPredicate.validate(mtw) )
                        continue;

                idx++;
                mtw.setChildIndex(idx);
                if( idx==0 )mtw.setFirstChild(true);

                nodesQueue.addFirst( mtw );
            }

            if( mtw!=null )mtw.setLastChild(true);
        }

//        if( mtw.

        return result;
    }

    private TreeWalk<T> deepParentChildBack()
    {
        if( nodesQueue.isEmpty() )return null;

        MutableTreeWalk<T> result = nodesQueue.poll();
        T currentNode = result.currentNode();
        int step = result.stepLevel();

        Iterable<T> nodes = extracter.extract(currentNode);
        if( nodes!=null )
        {
            int idx = -1;
            MutableTreeWalk<T> mtw = null;
            for( T n : nodes )
            {
                if( nodePredicate!=null )
                    if( !nodePredicate.validate(n) )
                        continue;

                mtw = 
                new MutableTreeWalk<T>(
                        result,
                        n,
                        result.currentLevel()+step,
                        step
                        );

                if( walkPredicate!=null )
                    if( !walkPredicate.validate(mtw) )
                        continue;

                idx++;
                mtw.setChildIndex(idx);
                if( idx==0 )mtw.setFirstChild(true);
                
                nodesQueue.addFirst( mtw );
            }

            if( mtw!=null )mtw.setLastChild(true);
        }

        return result;
    }

    private TreeWalk<T> topDown()
    {
        if( nodesQueue.isEmpty() )return null;

        MutableTreeWalk<T> result = nodesQueue.poll();
        T currentNode = result.currentNode();
//        T startNode = result.startNode();
        int step = result.stepLevel();
//        int startL = result.startLevel();

        Iterable<T> nodes = extracter.extract(currentNode);
        if( nodes!=null ){
            int idx = -1;
            MutableTreeWalk<T> mtw = null;

            for( T n : nodes )
            {
                if( nodePredicate!=null )
                    if( !nodePredicate.validate(n) )
                        continue;

                mtw = new MutableTreeWalk<T>(
                        result,
                        n,
                        result.currentLevel()+step,
                        step
                        );

                if( walkPredicate!=null )
                    if( !walkPredicate.validate(mtw) )
                        continue;

                idx++;
                mtw.setChildIndex(idx);
                if( idx==0 )mtw.setFirstChild(true);

                nodesQueue.add( mtw );
            }

            if( mtw!=null )mtw.setLastChild(true);
        }

//        if( visiter!=null ){
//            visiter.enter(result.currentNode());
//            visiter.exit(result.currentNode());
//        }

        return result;
    }

    @Override
    public void remove() {
//        throw new UnsupportedOperationException("Not supported yet.");
    }
}
