/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.list;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Queue;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.BasicPair;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.LockMethod;
import xyz.cofe.collection.Pair;
import xyz.cofe.common.GetListenersHelper;
import xyz.cofe.common.ListenersHelper;
import xyz.cofe.common.Reciver;

/**
 * Класс создающий уведомления об изменении списка. <br>
 * Не во всех случаях будет переда позиция изменяемого объекта, это связано стоб производительность не слишком падала при работе.<br>
 * Сообщение генериурются синхроно постфактум. <br>
 * Генерирует следующие сообщения:<br>
 * 
 * <table border='1' summary="Таблица событий">
 * <tr>
 * <td>Метод</td>
 * <td>Сообщение</td>
 * </tr>
 * 
 * <tr>
 * <td>add(E e)</td>
 * <td>AddedItemEvent&lt;E, EventList&lt;E&gt;&gt;</td>
 * </tr>
 * 
 * <tr>
 * <td>add(int index, E e)</td>
 * <td>AddedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;</td>
 * </tr>
 * 
 * <tr>
 * <td>
 * addAll(Collection&lt;? extends E&gt; c)
 * </td>
 * <td>AddedItemEvent&lt;E, EventList&lt;E&gt;&gt;</td>
 * </tr>
 * 
 * <tr>
 * <td>
 * addAll(int index, Collection&lt;? extends E&gt; c)
 * </td>
 * <td>AddedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;</td>
 * </tr>
 * 
 * <tr>
 * <td>
 * clear() 
 * </td>
 * <td>
 * RemovedPositionalItemEvent&lt;E, EventList&lt;E&gt;&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * remove(Object o) 
 * </td>
 * <td>
 * RemovedItemEvent&lt;E, EventList&lt;E&gt;&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * remove(int index) 
 * </td>
 * <td>
 * RemovedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * removeAll(Collection&lt;?&gt; c) 
 * </td>
 * <td>
 * RemovedItemEvent&lt;E, EventList&lt;E&gt;&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * retainAll(Collection&gt;?&gt; c)
 * </td>
 * <td>
 * RemovedPositionalItemEvent&lt;E, EventList&lt;E&gt;&gt;
 * </td>
 * </tr>
 * 
 * <tr>
 * <td>
 * set(int index, E element) 
 * </td>
 * <td>
 * UpdatedPositionalItemEvent&lt;E, EventList&lt;E&gt;,Integer&gt;
 * </td>
 * </tr>
 * 
 * 
 * </table>
 * @author Камнев Георгий Павлович
 * @param <E> Тип элемента коллеции
 */
public class BasicEventList<E> 
    extends ListWrapper<E> 
    implements EventList<E>, GetListenersHelper
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(BasicEventList.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }
    //</editor-fold>

    private static boolean eq( Object a, Object b ){
        if( a==null && b==null )return true;
        if( a==null && b!=null )return false;
        if( a!=null && b==null )return false;
        return a.equals(b);
    }
    
    public BasicEventList()
    {
        this(new ArrayList<E>());
    }

	/**
     * Конструктор.
     *
     * @param list
     *            Обвараичваемый список
     * @throws NullPointerException
     *             если list==null
     */
    public BasicEventList(List<E> list)
    {
        super(list);
    }

    //<editor-fold defaultstate="collapsed" desc="eventQueue - Очередь сообщений">
    /**
     * Очередь сообщений
     */
    protected final Queue<Object> eventQueue = new LinkedList<Object>();
    
    //<editor-fold defaultstate="collapsed" desc="addEventToQueue(e)">
    /**
     * Добавляет уведомления в очередь с учетом блокировки
     * @param event уведомления
     */
    protected void addEventToQueue(Object event)
    {
        synchronized(eventQueue){
            addEventToQueue0(event);
        }
    }
    
    /**
     * Добавляет уведомления в очередь без учета блокировки
     * @param job уведомления
     */
    private void addEventToQueue0(Object event){
        if( event!=null )eventQueue.add(event);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="fireQueueEvents()">
    /**
     * Выполняет уведомления из очереди с учетем блокировки
     */
    protected void fireQueueEvents(){
        synchronized(eventQueue){
            fireQueueEvents0();
        }
    }
    
    /**
     * Выполняет уведомления из очереди без учета блокировки
     */
    protected void fireQueueEvents0()
    {
        while(true)
        {
            Object event = eventQueue.poll();
            if( event==null )break;
            listeners.fireEvent(event);
        }
    }
    //</editor-fold>
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="Подписчики">
    protected final ListenersHelper<EventListListener, Object>
        listeners = new ListenersHelper<EventListListener, Object>(
            new Func2<Object, EventListListener, Object>()
        {
            @Override
            public Object apply(EventListListener l, Object e)
            {
                if( l!=null )l.listItemsChanged(e);
                return null;
            }
        }
    );

    @Override
    public ListenersHelper getListenersHelper() {
        return listeners;
    }
    
    @Override
    public Closeable addEventListListener(EventListListener<E> listener) {
        return listeners.addListener(listener);
    }
    
    @Override
    public Closeable addEventListListener(EventListListener<E> listener,boolean weak) {
        return listeners.addListener(listener, weak);
    }
    
    @Override
    public void removeEventListListener(EventListListener<E> listener) {
        listeners.removeListener(listener);
    }

    @Override
    public boolean containsEventListListener(EventListListener<E> listener) {
        return listeners.hasListener(listener);
    }
    
    /**
     * Вызываеться для отправки сообщения измения элементов списка
     * @param evnt Сообщение
     */
    protected void fireEventListMessage(Object evnt) {
        listeners.fireEvent(evnt);
    }
    
    /**
     * Добавляет подписчика изменений перед добавлением элемента
     * @param listener Подписчик
     * @return Обработчик закрытия прослушивания сообщений
     */
    @Override
    public Closeable onAdding(final Reciver<E> listener){
        if( listener==null )throw new IllegalArgumentException("listener==null");
        return addEventListListener(new SimpleListAdapter<E>(){
            @Override
            protected void adding(E e, EventList<E> list, Integer position) {
                if( listener!=null )listener.recive(e);
            }
        });
    }

    /**
     * Добавляет подписчика изменений добавления элемента
     * @param listener Подписчик
     * @return Обработчик закрытия прослушивания сообщений
     */
    @Override
    public Closeable onAdded(final Reciver<E> listener){
        if( listener==null )throw new IllegalArgumentException("listener==null");
        return addEventListListener(new SimpleListAdapter<E>(){
            @Override
            protected void added(E e, EventList<E> list, Integer position) {
                if( listener!=null )listener.recive(e);
            }
        });
    }
    
    /**
     * Добавляет подписчика изменений перед удадением элемента
     * @param listener Подписчик
     * @return Обработчик закрытия прослушивания сообщений
     */
    @Override
    public Closeable onRemoving(final Reciver<E> listener){
        if( listener==null )throw new IllegalArgumentException("listener==null");
        return addEventListListener(new SimpleListAdapter<E>(){
            @Override
            protected void removing(E e, EventList<E> list, Integer position) {
                if( listener!=null )listener.recive(e);
            }
        });
    }
    
    /**
     * Добавляет подписчика изменений удадения элемента
     * @param listener Подписчик
     * @return Обработчик закрытия прослушивания сообщений
     */
    @Override
    public Closeable onRemoved(final Reciver<E> listener){
        if( listener==null )throw new IllegalArgumentException("listener==null");
        return addEventListListener(new SimpleListAdapter<E>(){
            @Override
            protected void removed(E e, EventList<E> list, Integer position) {
                if( listener!=null )listener.recive(e);
            }
        });
    }

    @Override
    public Closeable onChanged(final Func3<Object, Integer, E, E> fn, boolean weak)
    {
        return addEventListListener(new EventListListener<E>()
            {
                @Override
                public void listItemsChanged(Object evnt)
                {
                    if( evnt instanceof UpdatedPositionalEvent ){
                        UpdatedPositionalEvent<E> updateEv = (UpdatedPositionalEvent)evnt;
                        Integer pRemove = updateEv.getRemovedItemPosition();
                        Integer pAdded  = updateEv.getAddedItemPosition();
                        if( !eq( pRemove,pAdded) ){
                            logWarning("update position event - removed pos({0}) != added pos({1})", 
                                       pRemove, pAdded);
                        }
                        E added = updateEv.getAddedItem();
                        E removed = updateEv.getRemovedItem();
                        if( fn!=null ){
                            fn.apply(pAdded, removed, added);
                        }
                    }else if( evnt instanceof AddedPositionalEvent ){
                        AddedPositionalEvent<E> ev = (AddedPositionalEvent)evnt;
                        Integer pos = ev.getAddedItemPosition();
                        if( pos==null ){
                            logWarning("added position event - added item pos({0}) is null", pos);
                        }
                        E eadded = ev.getAddedItem();
                        if( fn!=null ){
                            fn.apply(pos, null, eadded);
                        }
                    }else if( evnt instanceof AddedEvent ){
                        AddedEvent<E> ev = (AddedEvent)evnt;
                        E eadded = ev.getAddedItem();
                        if( fn!=null ){
                            fn.apply(null, null, eadded);
                        }
                    }else if( evnt instanceof RemovedPositionalEvent ){
                        RemovedPositionalEvent<E> ev = (RemovedPositionalEvent)evnt;
                        Integer pos = ev.getRemovedItemPosition();
                        if( pos==null ){
                            logWarning("removed position event - removed item pos({0}) is null", pos);
                        }
                        E edeleted = ev.getRemovedItem();
                        if( fn!=null ){
                            fn.apply(pos, edeleted, null );
                        }
                    }else if( evnt instanceof RemovedEvent ){
                        RemovedEvent<E> ev = (RemovedEvent)evnt;
                        E edeleted = ev.getRemovedItem();
                        if( fn!=null ){
                            fn.apply(null, edeleted, null);
                        }
                    }
                }
            },
            weak
        );
    }

    @Override
    public Closeable onChanged(Func3<Object, Integer, E, E> fn){
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return onChanged(fn, false);
    }

    @Override
    public Closeable onChanging(final Func3<Object, Integer, E, E> fn, boolean weak)
    {
        return addEventListListener(new EventListListener<E>()
            {
                @Override
                public void listItemsChanged(Object evnt)
                {
                    if( evnt instanceof UpdatingPositionalEvent ){
                        UpdatingPositionalEvent<E> updateEv = (UpdatingPositionalEvent)evnt;
                        Integer pRemove = updateEv.getRemovingItemPosition();
                        Integer pos  = updateEv.getAddingItemPosition();
                        if( !eq(pRemove,pos) ){
                            logWarning("updating position event - removing pos({0}) != added pos({1})", 
                                       pRemove, pos);
                        }
                        E adding = updateEv.getAddingItem();
                        E removing = updateEv.getRemovingItem();
                        if( fn!=null ){
                            fn.apply(pos, removing, adding);
                        }
                    }else if( evnt instanceof AddingPositionalEvent ){
                        AddingPositionalEvent<E> ev = (AddingPositionalEvent)evnt;
                        Integer pos = ev.getAddingItemPosition();
                        if( pos==null ){
                            logWarning("adding position event - adding item pos({0}) is null", pos);
                        }
                        E eadding = ev.getAddingItem();
                        if( fn!=null ){
                            fn.apply(pos, null, eadding);
                        }
                    }else if( evnt instanceof AddingEvent ){
                        AddingEvent<E> ev = (AddingEvent)evnt;
                        E eadding = ev.getAddingItem();
                        if( fn!=null ){
                            fn.apply(null, null, eadding);
                        }
                    }else if( evnt instanceof RemovingPositionalEvent ){
                        RemovingPositionalEvent<E> ev = (RemovingPositionalEvent)evnt;
                        Integer pos = ev.getRemovingItemPosition();
                        if( pos==null ){
                            logWarning("removing position event - removing item pos({0}) is null", pos);
                        }
                        E edeleting = ev.getRemovingItem();
                        if( fn!=null ){
                            fn.apply(pos, edeleting, null );
                        }
                    }else if( evnt instanceof RemovingEvent ){
                        RemovingEvent<E> ev = (RemovingEvent)evnt;
                        E edeleting = ev.getRemovingItem();
                        if( fn!=null ){
                            fn.apply(null, edeleting, null);
                        }
                    }
                }
            },
            weak
        );
    }
    
    @Override
    public Closeable onChanging(final Func3<Object, Integer, E, E> fn){
        return onChanging(fn, false);
    }
    
    //<editor-fold defaultstate="collapsed" desc="onUpdated()/onUpdating()">
    @Override
    public Closeable onUpdated(final Func3<Object, Integer, E, E> fn, boolean weak)
    {
        return addEventListListener(new EventListListener<E>()
        {
            @Override
            public void listItemsChanged(Object evnt)
            {
                if( evnt instanceof UpdatedPositionalEvent ){
                    UpdatedPositionalEvent<E> updateEv = (UpdatedPositionalEvent)evnt;
                    Integer pRemove = updateEv.getRemovedItemPosition();
                    Integer pAdded  = updateEv.getAddedItemPosition();
                    if( !eq(pRemove,pAdded) ){
                        logWarning("update position event - removed pos({0}) != added pos({1})",
                            pRemove, pAdded);
                    }
                    E added = updateEv.getAddedItem();
                    E removed = updateEv.getRemovedItem();
                    if( fn!=null ){
                        fn.apply(pAdded, removed, added);
                    }
                }
            }
        },
            weak
        );
    }
    
    @Override
    public Closeable onUpdated(Func3<Object, Integer, E, E> fn)
    {
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return onUpdated(fn, false);
    }
    
    @Override
    public Closeable onUpdating(final Func3<Object, Integer, E, E> fn, boolean weak)
    {
        return addEventListListener(new EventListListener<E>()
        {
            @Override
            public void listItemsChanged(Object evnt)
            {
                if( evnt instanceof UpdatingPositionalEvent ){
                    UpdatingPositionalEvent<E> updateEv = (UpdatingPositionalEvent)evnt;
                    Integer pRemove = updateEv.getRemovingItemPosition();
                    Integer pAdded  = updateEv.getAddingItemPosition();
                    if( !eq(pRemove,pAdded) ){
                        logWarning("updating position event - removing pos({0}) != adding pos({1})",
                            pRemove, pAdded);
                    }
                    E adding = updateEv.getAddingItem();
                    E removing = updateEv.getRemovingItem();
                    if( fn!=null ){
                        fn.apply(pAdded, removing, adding);
                    }
                }
            }
        },
            weak
        );
    }
    
    @Override
    public Closeable onUpdating(final Func3<Object, Integer, E, E> fn)
    {
        return onUpdating(fn, false);
    }
//</editor-fold>
    
    @Override
    public Closeable onInserted(final Func3<Object, Integer, E, E> fn){
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return onInserted(fn, false);
    }
    
    
    @Override
    public Closeable onInserted(final Func3<Object, Integer, E, E> fn, boolean weak)
    {
        return addEventListListener(new EventListListener<E>()
            {
                @Override
                public void listItemsChanged(Object evnt)
                {
                    if( evnt instanceof AddedPositionalEvent ){
                        AddedPositionalEvent<E> ev = (AddedPositionalEvent)evnt;
                        Integer pos = ev.getAddedItemPosition();
                        if( pos==null ){
                            logWarning("added position event - added item pos({0}) is null", pos);
                        }
                        E eAdded = ev.getAddedItem();
                        if( fn!=null ){
                            fn.apply(pos, null, eAdded);
                        }
                    }else if( evnt instanceof AddedEvent ){
                        AddedEvent<E> ev = (AddedEvent)evnt;
                        E eAdded = ev.getAddedItem();
                        if( fn!=null ){
                            fn.apply(null, null, eAdded);
                        }
                    }
                }
            },
            weak
        );
    }
    
    @Override
    public Closeable onInserting(final Func3<Object, Integer, E, E> fn, boolean weak)
    {
        return addEventListListener(new EventListListener<E>()
            {
                @Override
                public void listItemsChanged(Object evnt)
                {
                    if( evnt instanceof AddingPositionalEvent ){
                        AddingPositionalEvent<E> ev = (AddingPositionalEvent)evnt;
                        Integer pos = ev.getAddingItemPosition();
                        if( pos==null ){
                            logWarning("adding position event - adding item pos({0}) is null", pos);
                        }
                        E eadding = ev.getAddingItem();
                        if( fn!=null ){
                            fn.apply(pos, null, eadding);
                        }
                    }else if( evnt instanceof AddingEvent ){
                        AddingEvent<E> ev = (AddingEvent)evnt;
                        E eadding = ev.getAddingItem();
                        if( fn!=null ){
                            fn.apply(null, null, eadding);
                        }
                    }
                }
            },
            weak
        );
    }
    
    @Override
    public Closeable onInserting(final Func3<Object, Integer, E, E> fn){
        return onInserting(fn, false);
    }
    
    //<editor-fold defaultstate="collapsed" desc="onDeleting()/onDeleted()">
    @Override
    public Closeable onDeleted(final Func3<Object, Integer, E, E> fn){
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return onDeleted(fn, false);
    }
    
    @Override
    public Closeable onDeleted(final Func3<Object, Integer, E, E> fn, boolean weak)
    {
        return addEventListListener(new EventListListener<E>()
        {
            @Override
            public void listItemsChanged(Object evnt)
            {
                if( evnt instanceof RemovedPositionalEvent ){
                    RemovedPositionalEvent<E> ev = (RemovedPositionalEvent)evnt;
                    Integer pos = ev.getRemovedItemPosition();
                    if( pos==null ){
                        logWarning("removed position event - removed item pos({0}) is null", pos);
                    }
                    E eDeleted = ev.getRemovedItem();
                    if( fn!=null ){
                        fn.apply(pos, eDeleted, null );
                    }
                }else if( evnt instanceof RemovedEvent ){
                    RemovedEvent<E> ev = (RemovedEvent)evnt;
                    E eDeleted = ev.getRemovedItem();
                    if( fn!=null ){
                        fn.apply(null, eDeleted, null);
                    }
                }
            }
        },
            weak
        );
    }
    
    @Override
    public Closeable onDeleting(final Func3<Object, Integer, E, E> fn, boolean weak)
    {
        return addEventListListener(new EventListListener<E>()
        {
            @Override
            public void listItemsChanged(Object evnt)
            {
                if( evnt instanceof RemovingPositionalEvent ){
                    RemovingPositionalEvent<E> ev = (RemovingPositionalEvent)evnt;
                    Integer pos = ev.getRemovingItemPosition();
                    if( pos==null ){
                        logWarning("removing position event - removing item pos({0}) is null", pos);
                    }
                    E edeleting = ev.getRemovingItem();
                    if( fn!=null ){
                        fn.apply(pos, edeleting, null );
                    }
                }else if( evnt instanceof RemovingEvent ){
                    RemovingEvent<E> ev = (RemovingEvent)evnt;
                    E edeleting = ev.getRemovingItem();
                    if( fn!=null ){
                        fn.apply(null, edeleting, null);
                    }
                }
            }
        },
            weak
        );
    }
    
    @Override
    public Closeable onDeleting(final Func3<Object, Integer, E, E> fn){
        return onDeleting(fn, false);
    }
//</editor-fold>
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="lockRun(fun):Object">
    /**
     * Выполнение кода в блокировке
     * @param run код
     * @return Возвращенное значение из кода или null
     */
    protected Object lockRun( Func0 run ){
        if( run==null )throw new IllegalArgumentException("run==null");
        Object r = run.apply();
        return r;
    }

    /**
     * Выполнение кода в блокировке
     * @param run код
     * @param method метод map для которого вызывается блокировка
     * @return Возвращенное значение из кода или null
     */
    protected Object lockRun( Func0 run, LockMethod method){
        if( run==null )throw new IllegalArgumentException("run==null");
        return lockRun(run);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="add(elements)">
    /**
     * Добавление объекта в список. <br>
     * Генерирует сообщение <b>AddedItemEvent&lt;E, EventList&lt;E&gt;&gt;</b> для добавленого элемента
     * @param e Объект
     * @return Факт добавления
     */
    @Override
    public boolean add(final E e)
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return add0(e);
            }
        };
        
        boolean res = (boolean)(Boolean)
            lockRun(fn, new LockMethod("add", true));
        
        fireQueueEvents();
        
        return res;
    }
    
    protected boolean add0(E e){
        AddingEvent<E> insenv1 = new AddingEvent<E>(e,this);
        addEventToQueue(insenv1);
        fireQueueEvents();
        
        boolean res = super.add(e);
        
        if(res){
            AddedEvent<E> insenv2 = new AddedEvent<E>(e,this);
            addEventToQueue(insenv2);
        }
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="add(index,elements)">
    /**
     * Добавление объекта в список, в заданную позицию. <br>
     * Генерирует сообщение <b>AddedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;</b> для добавленого элемента
     * @param index Позиция
     * @param element Объект
     */
    @Override
    public void add(final int index, final E element) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                add0(index,element);
                return null;
            }
        };
        
        lockRun(fn, new LockMethod("add", true));
        
        fireQueueEvents();
    }
    
    protected void add0(int index, E element) {
        AddingPositionalEvent<E> insEvent1 = new AddingPositionalEvent<E>(element,this,index);
        addEventToQueue(insEvent1);
        fireQueueEvents();

        super.add(index, element);
        
        AddedPositionalEvent<E> insEvent2 = new AddedPositionalEvent<E>(element,this,index);
        addEventToQueue(insEvent2);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="addAll(elements)">
    /**
     * Добавление объектов в список. <br>
     * Генерирует сообщение <b>AddedItemEvent&lt;E, EventList&lt;E&gt;&gt;</b> для каждого добавленого элемента
     * @param c Объекты
     * @return Факт добавления
     */
    @Override
    public boolean addAll(final Collection<? extends E> c) {
        if( c==null )throw new IllegalArgumentException( "c==null" );
        
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return addAll0(c);
            }
        };
        
        boolean res = 
            (boolean)(Boolean)
            lockRun(fn, new LockMethod("addAll",true));
        
        fireQueueEvents();
        
        return res;
    }
    
    protected boolean addAll0(Collection<? extends E> c) {
        boolean res = false;
        for( E e : c ){
            boolean r = add0( e );
            res = r ? r : res;
        }
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="addAll(index,elements)">
    /**
     * Добавление объектов в список, в заданную позицию. <br>
     * Генерирует сообщение <b>AddedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;</b> для каждого добавленого элемента
     * @param index Позиция
     * @param c Объекты
     * @return Факт добавления
     */
    @Override
    public boolean addAll(final int index, final Collection<? extends E> c)
    {
        if( c==null )throw new IllegalArgumentException( "c==null" );
        
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return addAll0(index, c);
            }
        };
        
        boolean res = (boolean)(Boolean)
            lockRun(fn, new LockMethod("addAll", true));
        
        fireQueueEvents();
        
        return res;
    }
    
    protected boolean addAll0(int index, Collection<? extends E> c){
        boolean res = false;
        int idx = index;
        for( E e:c ){
            add0( idx, e );
            idx++;
            res = true;
        }
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="clear()">
    /**
     * Удаление всех объектов из списка. <br>
     * 
     * Сначала: <br>
     * 
     * Генерирует сообщение <b>RemovingPositionalItemEvent&lt;E, EventList&lt;E&gt;&gt;</b> 
     * с конца к началу списка для каждого элемента. <br><br>
     *
     * Затем удаляет элементы и <br>
     * 
     * генерирует сообщение <b>RemovedPositionalItemEvent&lt;E, EventList&lt;E&gt;&gt;</b> 
     * с конца к началу списка для каждого элемента.
     */
    @SuppressWarnings("unchecked")
    @Override
    public void clear()
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                clear0();
                return null;
            }
        };
        
        lockRun(fn, new LockMethod("clear", true));
        
        fireQueueEvents();
    }
    
    protected void clear0(){
        Object[] items = super.toArray();

        for( int i=items.length-1; i>=0; i-- ){
            addEventToQueue(
                // TODO Потенциальная ошибка - нет явного приведения типа
                new RemovingPositionalEvent(items[i], this, i)
            );
            fireQueueEvents();
        }
        
        super.clear();
        
        for( int i=items.length-1; i>=0; i-- ){
            addEventToQueue(
                // TODO Потенциальная ошибка - нет явного приведения типа
                new RemovedPositionalEvent(items[i], this, i)
            );
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="remove(element)">
    /**
     * Удаление объекта из списка. <br>
     * Для удаленного элемента генерирует сообщение <b>RemovedItemEvent&lt;E, EventList&lt;E&gt;&gt;</b>
     * @param o Объект
     * @return Факт удаления
     * @see xyz.cofe.collection.RemovedItemEvent
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean remove(final Object o)
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return remove0(o);
            }
        };
        
        boolean res = (boolean)(Boolean)
            lockRun(fn, new LockMethod("remove", true));

        fireQueueEvents();
        
        return res;
    }
    
    protected boolean remove0(Object o){
        if( super.contains(o) ){
            addEventToQueue(new RemovingEvent(o, this) );
            fireQueueEvents();
        }
        
        @SuppressWarnings("element-type-mismatch")
        boolean res = super.remove(o);

        if (res){
            addEventToQueue(new RemovedEvent(o, this) );
        }
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="remove(index)">
    /**
     * Удаление объекта из списка. <br>
     * Для удаленного элемента генерирует сообщение <b>RemovedPositionalItemEvent&lt;E, EventList&lt;E&gt;, Integer&gt;</b>
     * @param index Индекс элемента
     * @return Удаленный элемент
     * @see xyz.cofe.collection.RemovedPositionalItemEvent
     */
    @Override
    public E remove(final int index)
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return remove0(index);
            }
        };
        
        E item = (E)
            lockRun(fn, new LockMethod("remove", true));
        
        fireQueueEvents();
        return item;
    }
    
    protected E remove0(int index){
        addEventToQueue(new RemovingPositionalEvent<E>(get(index), this, index));
        fireQueueEvents();
        
        E item = super.remove(index);
        
        addEventToQueue(new RemovedPositionalEvent<E>(item, this, index));
        return item;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="removeAll(elements)">
    /**
     * Удаляет группу указанных объектов из списка. <br>
     * Вызывает метод remove(e) <br>
     * Генерирует <b>RemovedItemEvent&lt;E, EventList&lt;E&gt;&gt;</b> для удаленных объектов.
     * @param c Группа объектов
     * @return Факт удаления
     * @see #remove(java.lang.Object)
     * @see xyz.cofe.collection.RemovedItemEvent
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean removeAll(final Collection<?> c)
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return removeAll0(c);
            }
        };
        
        boolean modified = (boolean)(Boolean)
            lockRun(fn, new LockMethod("removeAll", true));
        
        fireQueueEvents();
        
        return modified;
    }
    
    protected boolean removeAll0(final Collection<?> c){
        int cnt = 0;
        for( Object o : c ){
            while( true ){
                if( remove0(o) ){
                    cnt++;
                }else{
                    break;
                }
            }
        }
        return cnt > 0;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="retainAll(elements)">
    /**
     * Удаляет объекты из списка, не входящиие в указанную группу объектов. <br>
     * <br>
     * Сравниваются исходная последовательность элементов, и та что осталась после удаления объектов. <br>
     * Те что были удалены для них генерирует сообщения <b>RemovedPositionalItemEvent&lt;E, EventList&lt;E&gt;&gt;</b>
     * в обратном порядке следования исходного списка элементов.
     * @param c Группа объектов
     * @return Факт удаления
     * @see xyz.cofe.collection.RemovedPositionalItemEvent
     */
    @SuppressWarnings("unchecked")
    @Override
    public boolean retainAll(final Collection<?> c)
    {
        if( c==null ){
            throw new IllegalArgumentException("c == null");
        }
        
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return retainAll0(c);
            }
        };
        
        boolean res = (boolean)(Boolean)
            lockRun(fn, new LockMethod("retainAll", true));
        
        fireQueueEvents();
        
        return res;
    }
    
    protected boolean retainAll0(Collection<?> c){
        // TODO что-то придумать чтоб получить линейный или хотябы логарифм сложность
        Object[] beforeItems = super.toArray();
        
        for( int i=0; i<beforeItems.length; i++ ){
            Object beforeItem = beforeItems[i];
            if( ! c.contains( beforeItem ) ){
                addEventToQueue(new RemovingPositionalEvent(beforeItem, this, i));
                fireQueueEvents();
            }
        }
        
        boolean res = super.retainAll(c);
        
        Object[] afterItems = super.toArray();
        
        if (res)
        {
            // Здесь обратная сортировака
            TreeSet<Pair<Object,Integer>> dels = new TreeSet<Pair<Object, Integer>>(
                    new Comparator<Pair<Object,Integer>>() {
                    @Override
                    public int compare(Pair<Object, Integer> o1, Pair<Object, Integer> o2) {
                        return -o1.B().compareTo(o2.B());
                    }
                }
            );
            
            int biIndex = -1;
            for (Object bi : beforeItems) {
                biIndex++;
                boolean found = false;
                for (Object ai : afterItems) {
                    if (bi == ai) {
                        found = true;
                        break;
                    }
                }
                
                if (!found) {
                    dels.add(new BasicPair<Object, Integer>(bi, biIndex));
                }
            }
            
            for (Pair<Object,Integer> delObj : dels)
            {
                // addEventToQueue(new RemovingPositionalEvent(delObj.A(), this, delObj.B() ) );

                // TODO Потенциальная ошибка - нет явного приведения типа
                addEventToQueue(new RemovedPositionalEvent(delObj.A(), this, delObj.B() ) );
            }
        }
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="set(index,element)">
    /**
     * Заменяет объект в указанной позиции. <br>
     * Генерирует сообщение <b>UpdatedPositionalItemEvent&lt;E, EventList&lt;E&gt;,Integer&gt;</b>
     * @param index Позиция элемента
     * @param element Новый элемент
     * @return Старый элемент
     * @see xyz.cofe.collection.UpdatedPositionalItemEvent
     */
    @Override
    public E set(final int index,final E element)
    {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return set0(index,element);
            }
        };
        
        E deleted = (E)lockRun(fn, new LockMethod("set", true));
        
        fireQueueEvents();
        
        return deleted;
    }
    
    protected E set0(int index, E element){
        E exists = super.get(index);

        addEventToQueue(new UpdatingPositionalEvent(exists, element, this, index));
        fireQueueEvents();
        
//        addEventToQueue(new RemovingPositionalEvent(exists, this, index));
//        fireQueueEvents();
//        
//        AddingPositionalEvent<E> insEvent1 = new AddingPositionalEvent<E>(element,this,index);
//        addEventToQueue(insEvent1);
//        fireQueueEvents();
        
        E deleted = super.set(index, element);
        
        addEventToQueue(new UpdatedPositionalEvent<E>(deleted, element, this, index) );
        return deleted;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="iterator()">
    @Override
    public Iterator<E> iterator() {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return iterator0();
            }
        };
        
        Iterator<E> itr = (Iterator<E>)lockRun(fn, new LockMethod("iterator", false));
        
        fireQueueEvents();
        
        return itr;
    }

    protected Iterator<E> iterator0(){
        final BasicEventList<E> f_elist = this;
        final Iterator<E> f_itr = getWrappedList().iterator();        
        return new BasicIterator<E>(f_elist, f_itr);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="listIterator()">
    @Override
    public ListIterator<E> listIterator() {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return listIterator0();
            }
        };
        ListIterator itr = (ListIterator)lockRun(fn, new LockMethod("listIterator", false));
        fireQueueEvents();
        return itr;
    }

    protected ListIterator<E> listIterator0() {
        return new BasicEventListIterator<E>(this);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="listIterator(index)">
    @Override
    public ListIterator<E> listIterator(final int index) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return listIterator0(index);
            }
        };
        
        ListIterator li = (ListIterator)
            lockRun(fn, new LockMethod("listIterator", false));
        
        fireQueueEvents();
        
        return li;
    }
    
    protected ListIterator<E> listIterator0(int index) {
        return new BasicEventListIterator<E>(this, index);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="subList">
    @Override
    public List<E> subList(final int fromIndex, final int toIndexExclusive) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return subList0(fromIndex,toIndexExclusive);
            }
        };
        
        List li = (List)
            lockRun(fn, new LockMethod("subList", false));
        
        fireQueueEvents();
        
        return li;
    }
    
    protected List<E> subList0(int fromIndex, int toIndexExclusive) {
        return new SubList<E>(this, fromIndex, toIndexExclusive);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="toArray()">
    @Override
    public <T> T[] toArray(final T[] a) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.toArray(a);
            }
        };
        
        T[] res = (T[])lockRun(fn, new LockMethod("toArray", false));
        
        return res;
    }
    
    @Override
    public Object[] toArray() {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.toArray();
            }
        };
        
        Object[] res = (Object[])lockRun(fn, new LockMethod("toArray", false));
        
        return res;
    }
//</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="size():int">
    @Override
    public int size() {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.size();
            }
        };
        
        int res = (int)(Integer)lockRun(fn, new LockMethod("size", false));
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="lastIndexOf(o):int">
    @Override
    public int lastIndexOf(final Object o) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.lastIndexOf(o);
            }
        };
        
        int res = (int)(Integer)lockRun(fn, new LockMethod("lastIndexOf", false));
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="isEmpty():boolean">
    @Override
    public boolean isEmpty() {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.isEmpty();
            }
        };
        
        boolean res = (boolean)(Boolean)lockRun(fn, new LockMethod("isEmpty", false));
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="indexOf(o):int">
    @Override
    public int indexOf(final Object o) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.indexOf(o);
            }
        };
        
        int res = (int)(Integer)lockRun(fn, new LockMethod("indexOf", false));
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="get(idx):E">
    @Override
    public E get(final int index) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.get(index);
            }
        };
        
        E res = (E)lockRun(fn, new LockMethod("get", false));
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="containsAll(c):boolean">
    @Override
    public boolean containsAll(final Collection<?> c) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.containsAll(c);
            }
        };
        
        boolean res = (boolean)(Boolean)lockRun(fn, new LockMethod("containsAll", false));
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="contains(o):boolean">
    @Override
    public boolean contains(final Object o) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.contains(o);
            }
        };
        
        boolean res = (boolean)(Boolean)lockRun(fn, new LockMethod("contains", false));
        
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="setWrappedList(l)">
    @Override
    protected synchronized void setWrappedList(final List<E> list) {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                BasicEventList.super.setWrappedList(list);
                return null;
            }
        };
        
        lockRun(fn, new LockMethod("setWrappedList", true));
        
        fireQueueEvents();
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getWrappedList():List">
    @Override
    public synchronized List<E> getWrappedList() {
        Func0 fn = new Func0() {
            @Override
            public Object apply() {
                return BasicEventList.super.getWrappedList();
            }
        };
        
        List res = (List)lockRun(fn, new LockMethod("getWrappedList", false));
        
        return res;
    }
    //</editor-fold>
}
