/*
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.collection.list;


import java.io.Closeable;
import java.io.IOException;
import java.util.ListIterator;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Интератор по списку
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 * @param <E> Тип элемента списка
 */
public class BasicEventListIterator<E> 
implements ListIterator<E>, Closeable
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(BasicEventListIterator.class.getName()).log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        Logger.getLogger(BasicEventListIterator.class.getName()).log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        Logger.getLogger(BasicEventListIterator.class.getName()).log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        Logger.getLogger(BasicEventListIterator.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(BasicEventListIterator.class.getName()).log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        Logger.getLogger(BasicEventListIterator.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(BasicEventListIterator.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    protected BasicEventList<E> elist = null;
    protected int position = -1;
    protected boolean closed = false;
    
    public BasicEventListIterator(BasicEventList<E> elist){
        if( elist==null )throw new IllegalArgumentException( "elist==null" );
        this.elist = elist;
        this.position = getBeginPosition();
    }

    public BasicEventListIterator(BasicEventList<E> elist,int initialIndex){
        if( elist==null )throw new IllegalArgumentException( "elist==null" );
        this.elist = elist;
        this.position = initialIndex;
        checkPositionBounds();
    }

    @Override
    public boolean hasNext() {
        if( closed )return false;
        int nidx = nextIndex();
        return indexInList(nidx);
    }

    @Override
    public E next() {
        if( closed )return null;
        this.position = nextIndex();
        if( indexInList(this.position) ){
            return elist.get(position);
        }
        return null;
    }

    @Override
    public boolean hasPrevious() {
        if( closed )return false;
        int nidx = previousIndex();
        return indexInList(nidx);
    }

    @Override
    public E previous() {
        if( closed )return null;
        this.position = previousIndex();
        if( indexInList(this.position) ){
            return elist.get(position);
        }
        return null;
    }
    
    protected void checkPositionBounds(){
        if( closed )return;
        int nmin = getBeginPosition();
        int nmax = getEndPosition();
        if( position>nmax && isLimitNextPosition() )position = nmax;
        if( position<nmin && isLimitPreviousPosition() )position = nmin;
    }
    
    protected boolean positionInBounds(int idx){
        if( closed )return false;
        if( idx<=getBeginPosition() )return false;
        if( idx>=getEndPosition() )return false;
        return true;
    }

    protected boolean indexInList(int idx){
        if( closed )return false;
        if( idx<0 )return false;
        if( idx>=elist.size() )return false;
        return true;
    }
    
    protected int getEndPosition(){
        if( closed )return -100;
        return elist.size();
    }
    
    protected boolean isLimitNextPosition(){ return true; }
    
    @Override
    public int nextIndex() {
        if( closed )return -100;
        int npos = position+1;
        int endidx = getEndPosition();
        if( npos>endidx && isLimitNextPosition() )npos = endidx;
        return npos;
    }
    
    protected int getBeginPosition(){
        if( closed )return -100;
        return -1;
    }
    
    protected boolean isLimitPreviousPosition(){ return true; }

    @Override
    public int previousIndex() {
        if( closed )return -100;
        int npos = position-1;
        int bidx = getBeginPosition();
        if( npos<bidx && isLimitPreviousPosition() )npos = bidx;
        return npos;
    }

    @Override
    public void remove() {
        if( closed )return;
        if( indexInList(position) ){
            elist.remove(position);
        }
        checkPositionBounds();
    }

    @Override
    public void set(E e) {
        if( closed )return;
        if( indexInList(position) ){
            elist.set(position, e);
        }
    }
    
    protected int getFirstIndex(){ return 0; }

    @Override
    public void add(E e) {
        if( closed )return;
        if( indexInList(position) ){
            elist.add(position,e);
        }else if( position<=getBeginPosition() ){
            int fidx = getFirstIndex();
            elist.add(fidx,e);
            position = fidx;
        }else if( position>=getEndPosition() ){
            elist.add(e);
            position = getEndPosition();
        }
    }

    @Override
    public void close() throws IOException {
        if( closed )return;
        this.closed = true;
        if( this.elist!=null ){
            this.elist = null;
        }
    }
}
