/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.list;

import java.io.Closeable;
import xyz.cofe.collection.Func3;
import xyz.cofe.common.Reciver;

/**
 * Добавление/удаление слушателей списка
 * @author Kamnev Georgiy (nt.gocha@gmail.com).
 * @param <E> Тип списка
 */
public interface EventListSender<E>
{
	/**
     * Добавляет слушателя на реакцию изменения списка
     * @param listener слушатель
     * @return Обработчик отписки слушателя от прослушки
     */
    public abstract Closeable addEventListListener(EventListListener<E> listener);

	/**
     * Добавляет слушателя на реакцию изменения списка
     * @param listener слушатель
     * @param weakListener прослушиватель вешается на weak ссылку
     * @return Обработчик отписки слушателя от прослушки
     */
    public abstract Closeable addEventListListener(EventListListener<E> listener,boolean weakListener);
    
    /**
     * Удаляет слушателья с реакции изменения списка.
     * @param listener слушатель
     */
    public abstract void removeEventListListener(EventListListener<E> listener);

    /**
     * Проверяет что указанный подписчик прослушивает изменения
     * @param listener Подписчик
     * @return true - подписчик подписан на события
     */
    boolean containsEventListListener(EventListListener<E> listener);
    
    /**
     * Добавляет подписчика изменений перед добавлением элемента
     * @param listener Подписчик
     * @return Обработчик закрытия прослушивания сообщений
     */
    public Closeable onAdding(final Reciver<E> listener);
    
    /**
     * Добавляет подписчика изменений после добавления элемента
     * @param listener Подписчик
     * @return Обработчик закрытия прослушивания сообщений
     */
    public Closeable onAdded(final Reciver<E> listener);
    
    /**
     * Добавляет подписчика изменений перед удадением элемента
     * @param listener Подписчик
     * @return Обработчик закрытия прослушивания сообщений
     */
    public Closeable onRemoving(final Reciver<E> listener);

    /**
     * Добавляет подписчика изменений после удадения элемента
     * @param listener Подписчик
     * @return Обработчик закрытия прослушивания сообщений
     */
    public Closeable onRemoved(final Reciver<E> listener);
    
    //<editor-fold defaultstate="collapsed" desc="onChanging()/onChanged()">
    /**
     * Добавляет подписчика на изменения
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в карте обновленно.
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в карту.
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из карты.
     * <br>
     *
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onChanged(final Func3<Object,Integer,E,E> fn,boolean weak);
    
    /**
     * Добавляет подписчика на начало изменений
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в карте обновленно.
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в карту.
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из карты.
     * <br>
     *
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onChanging(final Func3<Object,Integer,E,E> fn,boolean weak);
    
    /**
     * Добавляет подписчика на начало изменений
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в карте обновленно.
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в карту.
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из карты.
     * <br>
     *
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onChanging(final Func3<Object,Integer,E,E> fn);
    
    /**
     * Добавляет подписчика на изменения
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в карте обновленно.
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в карту.
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из карты.
     * <br>
     *
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onChanged(final Func3<Object,Integer,E,E> fn);
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="onUpdating()/onUpdated()">
    /**
     * Добавляет подписчика только на изменение значения в списке, не добавление, ни удаление
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в списке обновленно.
     * <br>
     * <br>
     *
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onUpdating(final Func3<Object,Integer,E,E> fn,boolean weak);
    
    /**
     * Добавляет подписчика только на изменение значения в списке, не добавление, ни удаление
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в списке обновленно.
     * <br>
     * <br>
     *
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onUpdating(final Func3<Object,Integer,E,E> fn);
    
    /**
     * Добавляет подписчика только на изменение значения в списке, не добавление, ни удаление
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в списке обновленно.
     * <br>
     * <br>
     *
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onUpdated(final Func3<Object,Integer,E,E> fn,boolean weak);
    
    /**
     * Добавляет подписчика только на изменение значения в списке, не добавление, ни удаление
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в списке обновленно.
     * <br>
     * <br>
     *
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onUpdated(final Func3<Object,Integer,E,E> fn);
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="onInserting()/onInserted()">
    /**
     * Добавляет подписчика только на начало добавления в список
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в список.
     * <br>
     * <br>
     *
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onInserting(final Func3<Object,Integer,E,E> fn,boolean weak);
    
    /**
     * Добавляет подписчика только на начало добавления в список
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в список.
     * <br>
     * <br>
     *
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onInserting(final Func3<Object,Integer,E,E> fn);
    
    /**
     * Добавляет подписчика только на добавление в список
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в список.
     * <br>
     * <br>
     *
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onInserted(final Func3<Object,Integer,E,E> fn,boolean weak);
    
    /**
     * Добавляет подписчика только на добавление в список
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в список.
     * <br>
     * <br>
     *
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onInserted(final Func3<Object,Integer,E,E> fn);
//</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="onDeleting()/onDeleted()">
    /**
     * Добавляет подписчика только на удаление из списка
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из списка.
     * <br>
     * <br>
     *
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onDeleted(final Func3<Object,Integer,E,E> fn,boolean weak);
    
    /**
     * Добавляет подписчика только на удаление из списка
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из списка.
     * <br>
     * <br>
     *
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onDeleted(final Func3<Object,Integer,E,E> fn);
    
    /**
     * Добавляет подписчика только на начало удаления из списка
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из списка.
     * <br>
     * <br>
     *
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onDeleting(final Func3<Object,Integer,E,E> fn,boolean weak);
    
    /**
     * Добавляет подписчика только на начало удаления из списка
     * @param fn
     * <span style='font-family:monospace'>
     * Функция(Key <b>ключ</b>, Value <b>староеЗначение</b>, Value <b>новоеЗначение</b>):Object
     * </span>
     * <br>
     * <br>
     *
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из списка.
     * <br>
     * <br>
     *
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onDeleting(final Func3<Object,Integer,E,E> fn);
//</editor-fold>
}
