/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.map;

//import com.sun.xml.internal.ws.Closeable;
//import java.util.HashMap;
import java.io.Closeable;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.Func2;
import xyz.cofe.collection.Func3;
import xyz.cofe.collection.LockMethod;
import xyz.cofe.common.GetListenersHelper;
import xyz.cofe.common.ListenersHelper;

/**
 * Базовая карта с поддержкой событий измений данных карты
 * @author gocha
 * @param <Key> Тип ключа
 * @param <Value> Тип значения
 */
public class BasicEventMap<Key,Value> 
    extends MapWrapper<Key,Value> 
    implements EventMap<Key,Value>, GetListenersHelper
{
    //<editor-fold defaultstate="collapsed" desc="BasicEventMap()">
    /**
     * Конструктор по умолчанию
     */
    public BasicEventMap()
    {
        this( new LinkedHashMap<Key,Value>() );
    }
    
    /**
     * Конструктор
     * @param wrappedMap Исходная карта
     */
    public BasicEventMap(Map<Key,Value> wrappedMap)
    {
        super(wrappedMap);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getListenersLock()">
    /**
     * Возвращает блокировку на listeners методы
     * @return блокировка на методы listeners или null
     */
    public Lock getListenersLock(){
        return listeners.getLock();
    }
    //</editor-fold>

    protected boolean listenObjectChanged(){ return true; }

    //<editor-fold defaultstate="collapsed" desc="eventQueue - Очередь сообщений">
    /**
     * Очередь сообщений
     */
    protected final Queue<KeyValueMapEvent<Key,Value>> eventQueue = new LinkedList<KeyValueMapEvent<Key,Value>>();
    
    //<editor-fold defaultstate="collapsed" desc="addEventToQueue(e)">
    /**
     * Добавляет уведомления в очередь с учетом блокировки
     * @param event уведомления
     */
    protected void addEventToQueue(KeyValueMapEvent<Key,Value> event)
    {
        synchronized(eventQueue){
            addEventToQueue0(event);
        }
    }
    
    /**
     * Добавляет уведомления в очередь без учета блокировки
     * @param job уведомления
     */
    private void addEventToQueue0(KeyValueMapEvent<Key,Value> event){
        if( event!=null )eventQueue.add(event);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="fireQueueEvents()">
    /**
     * Выполняет уведомления из очереди с учетем блокировки
     */
    protected void fireQueueEvents(){
        synchronized(eventQueue){
            fireQueueEvents0();
        }
    }
    
    /**
     * Выполняет уведомления из очереди без учета блокировки
     */
    private void fireQueueEvents0()
    {
        while(true)
        {
            KeyValueMapEvent<Key,Value> event = eventQueue.poll();
            if( event==null )break;
            listeners.fireEvent(event);
        }
    }
    //</editor-fold>
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="listeners : ListenersHelper">
    protected final ListenersHelper<EventMapListener,MapEvent> listeners = new ListenersHelper<EventMapListener, MapEvent>(
        new Func2<Object, EventMapListener, MapEvent>() {
            @Override
            public Object apply(EventMapListener lst, MapEvent e) {
                lst.eventMap(e);
                return null;
            }
        });

    @Override
    public ListenersHelper getListenersHelper() {
        return listeners;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="addEventMapListener(l)">
    @Override
    public Closeable addEventMapListener(EventMapListener<Key, Value> listener) {
        return addEventMapListener(listener,false);
    }
    
    @Override
    public Closeable addEventMapListener(EventMapListener<Key, Value> listener, boolean weakRef) {
        return listeners.addListener(listener, weakRef);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="removeEventMapListener(l)">
    @Override
    public void removeEventMapListener(EventMapListener<Key, Value> listener) {
        listeners.removeListener(listener);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="containsEventMapListener(e)">
    @Override
    public boolean containsEventMapListener(EventMapListener<Key, Value> listener) {
        return listeners.hasListener(listener);
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="onChanged()">
    @Override
    public Closeable onChanged(final Func3<Object, ? super Key, ? super Value, ? super Value> fn) {
        return onChanged(fn, false);
    }
    
    @Override
    public Closeable onChanged(final Func3<Object, ? super Key, ? super Value, ? super Value> fn, boolean weak) {
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return addEventMapListener(new EventMapListener<Key, Value>() {
            @Override
            public void eventMap(MapEvent<Key, Value> event) {
                if( event instanceof UpdatedMapEvent ){
                    UpdatedMapEvent<Key, Value> e = (UpdatedMapEvent)event;
                    fn.apply(e.getKey(), e.getOldValue(), e.getValue());
                }else if( event instanceof InsertedMapEvent ){
                    InsertedMapEvent<Key, Value> inserted = (InsertedMapEvent)event;
                    fn.apply(inserted.getKey(), null, inserted.getValue());
                }else if( event instanceof DeletedMapEvent ){
                    DeletedMapEvent<Key, Value> deleted = (DeletedMapEvent)event;
                    fn.apply(deleted.getKey(), deleted.getValue(), null);
                }
            }
        });
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="onUpdated()">
    @Override
    public Closeable onUpdated(final Func3<Object, ? super Key, ? super Value, ? super Value> fn) {
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return onUpdated(fn, false);
    }
    
    @Override
    public Closeable onUpdated(final Func3<Object, ? super Key, ? super Value, ? super Value> fn, boolean weak) {
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return onChanged(new Func3<Object, Key, Value, Value>() {
            @Override
            public Object apply(Key k, Value oldv, Value newv) {
                if( newv!=null && oldv!=null )
                    fn.apply(k, oldv, newv);
                
                return null;
            }
        }, weak);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="onInserted()">
    @Override
    public Closeable onInserted(final Func3<Object, ? super Key, ? super Value, ? super Value> fn) {
        return onInserted(fn,false);
    }    
    
    @Override
    public Closeable onInserted(final Func3<Object, ? super Key, ? super Value, ? super Value> fn, boolean weak) {
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return onChanged(new Func3<Object, Key, Value, Value>() {
            @Override
            public Object apply(Key k, Value oldv, Value newv) {
                if( newv!=null && oldv==null )
                    fn.apply(k, oldv, newv);
                
                return null;
            }
        }, weak);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="onDeleted()">
    @Override
    public Closeable onDeleted(final Func3<Object,? super Key,? super Value,? super Value> fn){
        return onDeleted(fn, false);
    }
    
    @Override
    public Closeable onDeleted(final Func3<Object,? super Key,? super Value,? super Value> fn, boolean weak){
        if( fn==null )throw new IllegalArgumentException( "fn==null" );
        return onChanged(new Func3<Object, Key, Value, Value>() {
            @Override
            public Object apply(Key k, Value oldv, Value newv) {
                if( newv==null && oldv!=null )
                    fn.apply(k, oldv, newv);
                
                return null;
            }
        }, weak);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="fireEvent(e)">
    /**
     * Увеомляет подписчиков о сообщении
     * @param event Событие
     */
    protected void fireEvent(MapEvent event)
    {
        listeners.fireEvent(event);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="(fire) delete events">
    //<editor-fold defaultstate="collapsed" desc="fireDeleted(k,v)">
    /**
     * Сообщает о удалении ключа
     * @param k Ключ
     * @param v Значение
     */
    protected void fireDeleted(Key k,Value v)
    {
        fireEvent(createDeleted(k, v));
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="createDeleted(k,v)">
    /**
     * Создает уведомление о удалении
     * @param k ключ
     * @param v значение
     * @return уведомление
     */
    protected DeletedMapEvent<Key, Value> createDeleted(Key k,Value v){
        return new DeletedMapEvent<Key, Value>(this, k, v);
    }
    //</editor-fold>
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="(fire) insert events">
    //<editor-fold defaultstate="collapsed" desc="fireInserted(k,v)">
    /**
     * Сообщает при добавлении ключа
     * @param k Ключ
     * @param v Значение
     */
    protected void fireInserted(Key k,Value v)
    {
        fireEvent(createInserted(k, v));
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="createInserted(k,v)">
    /**
     * Создает уведомление о добавлении
     * @param k ключ
     * @param v значение
     * @return уведомление
     */
    protected InsertedMapEvent<Key, Value> createInserted(Key k,Value v){
        return new InsertedMapEvent<Key, Value>(this, k, v);
    }
    //</editor-fold>
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="(fire) update events">
    //<editor-fold defaultstate="collapsed" desc="fireUpdated(k,v,old)">
    /**
     * Сообщает о измении ключа
     * @param k Ключ
     * @param v Новое значение
     * @param old Старое значение
     */
    protected void fireUpdated(Key k,Value v,Value old)
    {
        fireEvent(createUpdated(k, v, old));
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="createUpdated(k,v,old)">
    /**
     * Создает уведомление о обновление
     * @param k ключ
     * @param v новое значение
     * @param old старое значение
     * @return уведомление
     */
    protected UpdatedMapEvent<Key, Value> createUpdated(Key k,Value v,Value old){
        return new UpdatedMapEvent<Key, Value>(this, k, v, old);
    }
    //</editor-fold>
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="lockRun(fun):Object">
    /**
     * Выполнение кода в блокировке
     * @param run код-функция
     * @return Результат функции
     */
    protected Object lockRun( Func0 run ){
        if( run==null )throw new IllegalArgumentException("run==null");
        Object r = run.apply();
        return r;
    }

    /**
     * Выполнение кода в блокировке
     * @param run код
     * @param method метод map для которого вызывается блокировка
     * @return Результат функции
     */
    protected Object lockRun( Func0 run, LockMethod method){
        if( run==null )throw new IllegalArgumentException("run==null");
        return lockRun(run);
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="clear()">
    @Override
    public void clear() {
        lockRun(
            new Func0() { @Override public Object apply() {
                clear0();
                return null;
            }
        }, new LockMethod("clear",true));
        fireQueueEvents();
    }
    
    private void clear0()
    {
        for( Key k : keySet() )
        {
            final Value v = get(k);
            final Key fk = k;
            addEventToQueue(createDeleted(fk, v));
        }
        super.clear();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="put(k,v)">
    @Override
    public Value put(final Key key,final Value value){
        Value r = (Value)lockRun(
            new Func0() { @Override public Object apply() {
                Value r = put0(key,value);
                return r;
            }
        }, new LockMethod("put",true));
        fireQueueEvents();
        return r;
    }
    
    // TODO v1 Протестировать что передает индекс в случаии Add
    // TODO v1 Протестировать что передает индекс в случаии Update
//    @Override
    private Value put0(Key key, Value value)
    {
        if( containsKey(key) )
        {
            final Key k = key;
            final Value vnew = value;
            final Value res = super.put(key, value);
            addEventToQueue(createUpdated(k, vnew, res));
            return res;
        }else{
            final Key k = key;
            final Value v = value;
            final Value res = super.put(key, value);
            addEventToQueue(createInserted(k, v));
            return res;
        }
    }
    //</editor-fold>

    private boolean throwExceptionOnNullArg(){ return true; }

    //<editor-fold defaultstate="collapsed" desc="putAll(m)">
    @Override
    public void putAll(final Map<? extends Key, ? extends Value> m)
    {
        lockRun(
            new Func0() { @Override public Object apply() {
                putAll0(m);
                return null;
            }
        }, new LockMethod("putAll",true));
        fireQueueEvents();
    }
    
    // TODO v1 Протестировать что передает индекс в случаии Add
    // TODO v1 Протестировать что передает индекс в случаии Update
//    @Override
    private void putAll0(Map<? extends Key, ? extends Value> m)
    {
        if (m == null) {
            if( throwExceptionOnNullArg() )
                throw new IllegalArgumentException("m == null");
            else
                return;
        }
        
        for( Key k : m.keySet() )
        {
            Value v = m.get(k);
            put0(k, v);
        }
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="remove(k)">
    @Override
    @SuppressWarnings("element-type-mismatch")
    public Value remove(final Object key){
        Value r = (Value)lockRun(new Func0() {
            @Override
            public Object apply() {
                return remove0(key);
            }
        }, new LockMethod("remove",true));
        fireQueueEvents();
        return r;
    }
    
    // TODO v1 Протестировать что передает индекс в случаии Remove
//    @Override
//    @SuppressWarnings("element-type-mismatch")
    @SuppressWarnings("element-type-mismatch")
    private Value remove0(Object key)
    {
        if( containsKey(key) )
        {
            try
            {
                final Value v = get(key);
                final Key k = (Key)key;
                addEventToQueue(createDeleted(k, v));
            }
            catch(ClassCastException ex)
            {
            }
        }
        
        Value res = super.remove(key);
        return res;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="setWrappedMap(m)">
    /**
     * Генерировать события Insert/Delete при вызове setWrappedMap
     * @return true (по умолч) - генерация событий.
     */
    protected boolean sendNewListOnNewWrapperMap()
    {
        return true;
    }
    
    /**
     * Устанавливает новую карту
     * @param wrappedMap карта
     */
    @Override
    public void setWrappedMap(final Map<Key, Value> wrappedMap)
    {
        lockRun(new Func0() {
            @Override
            public Object apply() {
                setWrappedMap0(wrappedMap);
                return null;
            }
        }, new LockMethod("setWrappedMap",true));
        fireQueueEvents();
    }
    
    private void setWrappedMap0(Map<Key, Value> wrappedMap)
    {
        if (wrappedMap == null) {
            if( throwExceptionOnNullArg() )
                throw new IllegalArgumentException("wrappedMap == null");
            else
                return;
        }
        
        if( sendNewListOnNewWrapperMap() )
        {
            for( Key k : keySet() )
            {
                final Value v = get(k);
                final Key fk = k;
                addEventToQueue(createDeleted(k, v));
            }
            
            for( Key key : wrappedMap.keySet() )
            {
                final Value v = get(key);
                final Key k = key;
                addEventToQueue(createInserted(k, v));
            }
        }
        
        super.setWrappedMap(wrappedMap);
    }
    //</editor-fold>
    
    //////////////////////////////////////////////////////////////////////////////////////////////////
    // read methods

    //<editor-fold defaultstate="collapsed" desc="isEmpty()">
    /* (non-Javadoc) @see MapWrapper */
    @Override
    public boolean isEmpty() {
        boolean res = (boolean)(Boolean)lockRun(
            new Func0() { @Override public Object apply() {
                return BasicEventMap.super.isEmpty();
            }}
            , new LockMethod("isEmpty",false)
        );
        return res;
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="containsKey(k)">
    /* (non-Javadoc) @see MapWrapper */
    @Override
    public boolean containsKey(final Object key) {
        boolean res = (boolean)(Boolean)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventMap.super.containsKey(key);
            }
        }, new LockMethod("containsKey",false));
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="containsValue(v)">
    /* (non-Javadoc) @see MapWrapper */
    @Override
    public boolean containsValue(final Object value) {
        boolean res = (boolean)(Boolean)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventMap.super.containsValue(value);
            }
        }, new LockMethod("containsValue",false));
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="entrySet()">
    /* (non-Javadoc) @see MapWrapper */
    @Override
    public Set<Entry<Key, Value>> entrySet() {
        Set<Entry<Key, Value>> res = (Set<Entry<Key, Value>>)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventMap.super.entrySet();
            }
        }, new LockMethod("entrySet",false));
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="get(k)">
    /* (non-Javadoc) @see MapWrapper */
    @Override
    public Value get(final Object key) {
        Value res = (Value)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventMap.super.get(key);
            }
        },new LockMethod("get",false));
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="keySet()">
    /* (non-Javadoc) @see MapWrapper */
    @Override
    public Set<Key> keySet() {
        Set<Key> res = (Set<Key>)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventMap.super.keySet();
            }
        },new LockMethod("keySet",false));
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="size()">
    /* (non-Javadoc) @see MapWrapper */
    @Override
    public int size() {
        int res = (int)(Integer)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventMap.super.size();
            }
        },new LockMethod("size",false));
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="values()">
    /* (non-Javadoc) @see MapWrapper */
    @Override
    public Collection<Value> values() {
        Collection<Value> res = (Collection<Value>)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventMap.super.values();
            }
        },new LockMethod("values",false));
        return res;
    }
    //</editor-fold>

    //<editor-fold defaultstate="collapsed" desc="getWrappedMap()">
    /* (non-Javadoc) @see MapWrapper */
    @Override
    public Map<Key, Value> getWrappedMap() {
        Map<Key, Value> res = (Map<Key, Value>)lockRun(new Func0() {
            @Override
            public Object apply() {
                return BasicEventMap.super.getWrappedMap();
            }
        },new LockMethod("getWrappedMap",false));
        return res;
    }
    //</editor-fold>
}
