/* 
 * The MIT License
 *
 * Copyright 2014 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */
package xyz.cofe.collection.map;

import java.io.Closeable;
import xyz.cofe.collection.Func3;

/**
 * Интерфес карты с извещениями о изменении
 * @author gocha
 * @param <Key> Ключ коллеции
 * @param <Value> Значение коллеции
 */
public interface EventMapSender<Key,Value>
{
    /**
     * Добавляет подписчика измений
     * @param listener Подписчик
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable addEventMapListener(EventMapListener<Key,Value> listener);

    /**
     * Добавляет подписчика измений
     * @param listener Подписчик
     * @param weakRef Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable addEventMapListener(EventMapListener<Key,Value> listener,boolean weakRef);
    
    /**
     * Удаляет подписчика измений
     * @param listener Подписчик
     */
    void removeEventMapListener(EventMapListener<Key,Value> listener);

    /**
     * Проверяет что указанный подписчик прослушивает изменения
     * @param listener Подписчик
     * @return true - подписчик подписан на события
     */
    boolean containsEventMapListener(EventMapListener<Key,Value> listener);
    
    /**
     * Добавляет подписчика только на добавление в карту
     * @param fn 
     * <span style='font-family:monospace'>
     * Функция(Key ключ, Value староеЗначение, Value новоеЗначение):Object
     * </span> 
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в карту.
     * <br>
     * <br>
     * 
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onInserted(final Func3<Object,? super Key,? super Value,? super Value> fn,boolean weak);
    
    /**
     * Добавляет подписчика только на добавление в карту
     * @param fn 
     * <span style='font-family:monospace'>
     * Функция(Key ключ, Value староеЗначение, Value новоеЗначение):Object
     * </span> 
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в карту.
     * <br>
     * <br>
     * 
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onInserted(final Func3<Object,? super Key,? super Value,? super Value> fn);
    
    /**
     * Добавляет подписчика только на удаление из карты
     * @param fn 
     * <span style='font-family:monospace'>
     * Функция(Key ключ, Value староеЗначение, Value новоеЗначение):Object
     * </span> 
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из карты.
     * <br>
     * <br>
     * 
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onDeleted(final Func3<Object,? super Key,? super Value,? super Value> fn,boolean weak);
    
    /**
     * Добавляет подписчика только на удаление из карты
     * @param fn 
     * <span style='font-family:monospace'>
     * Функция(Key ключ, Value староеЗначение, Value новоеЗначение):Object
     * </span> 
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из карты.
     * <br>
     * <br>
     * 
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onDeleted(final Func3<Object,? super Key,? super Value,? super Value> fn);
    
    /**
     * Добавляет подписчика на изменения
     * @param fn 
     * <span style='font-family:monospace'>
     * Функция(Key ключ, Value староеЗначение, Value новоеЗначение):Object
     * </span> 
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в карте обновленно.
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в карту.
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из карты.
     * <br>
     * 
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onChanged(final Func3<Object,? super Key,? super Value,? super Value> fn,boolean weak);
    
    /**
     * Добавляет подписчика на изменения
     * @param fn 
     * <span style='font-family:monospace'>
     * Функция(Key ключ, Value староеЗначение, Value новоеЗначение):Object
     * </span> 
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в карте обновленно.
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> != null и <i>староеЗначение</i> == null - добавлено значение в карту.
     * <br>
     * <br>
     * 
     * Если <i>новоеЗначение</i> == null и <i>староеЗначение</i> != null - удалено значение из карты.
     * <br>
     * 
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onChanged(final Func3<Object,? super Key,? super Value,? super Value> fn);
    
    /**
     * Добавляет подписчика только на изменение значения в карте, не добавление, ни удаление
     * @param fn 
     * <span style='font-family:monospace'>
     * Функция(Key ключ, Value староеЗначение, Value новоеЗначение):Object
     * </span> 
     * <br>
     * <br>
     * 
     * <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в карте обновленно.
     * <br>
     * <br>
     * 
     * @param weak Добавить подписчика как weak ссылку
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onUpdated(final Func3<Object,? super Key,? super Value,? super Value> fn,boolean weak);

    /**
     * Добавляет подписчика только на изменение значения в карте, не добавление, ни удаление
     * @param fn 
     * <span style='font-family:monospace'>
     * Функция(Key ключ, Value староеЗначение, Value новоеЗначение):Object
     * </span> 
     * <br>
     * <br>
     * 
     * <i>новоеЗначение</i> != null и <i>староеЗначение</i> != null - значение в карте обновленно.
     * <br>
     * <br>
     * 
     * @return Обработчик закрытия/отписки от уведомлений
     */
    Closeable onUpdated(final Func3<Object,? super Key,? super Value,? super Value> fn);
}
