/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy <nt.gocha@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.collection.map;

import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.LockMethod;

/**
 * Event map с поддержкой блокировки Lock
 * @author nt.gocha@gmail.com
 */
public class LockEventMap <Key,Value> extends BasicEventMap<Key,Value>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(LockEventMap.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(LockEventMap.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(LockEventMap.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(LockEventMap.class.getName(), method, result);
    }
    //</editor-fold>

    public LockEventMap() {
        this.lockWrite = createLock();
        this.lockRead = this.lockWrite;
    }

    public LockEventMap(Map<Key, Value> wrappedMap) {
        super(wrappedMap);
        this.lockWrite = createLock();
        this.lockRead = this.lockWrite;
    }

    public LockEventMap(Map<Key, Value> wrappedMap, Lock lock) {
        super(wrappedMap);
        this.lockWrite = lock==null ? createLock() : lock;
        this.lockRead = this.lockWrite;
    }

    public LockEventMap(Map<Key, Value> wrappedMap, Lock lockRead, Lock lockWrite) {
        super(wrappedMap);
        this.lockWrite = lockWrite!=null ? lockWrite : createLock();
        this.lockRead = lockRead!=null ? lockRead : this.lockWrite;
    }

    protected final Lock lockWrite;
    protected final Lock lockRead;

    //<editor-fold defaultstate="collapsed" desc="createLock()">
    /**
     * Создает блокировку ReentrantLock
     * @return блокировка
     */
    protected Lock createLock(){
        return new ReentrantLock();
    }
    //</editor-fold>
    
    //<editor-fold defaultstate="collapsed" desc="getLock()">
    /**
     * Возвращает блокировку на методы Map
     * @return блокировка на методы map
     */
    public Lock getLockWrite(){
        return lockWrite;
    }

    /**
     * Возвращает блокировку на методы Map
     * @return блокировка на методы map
     */
    public Lock getLockRead() {
        return lockRead;
    }
    //</editor-fold>
    
    @Override
    protected Object lockRun(Func0 run, LockMethod method) {
        if( lockWrite==lockRead ){
            return lockRun(run);
        }else{
            if( method!=null ){
                boolean isRead = method.getWrite()==null || !method.getWrite();
                boolean isWrite = !isRead;
                if( isWrite ){
                    // write Lock
                    return lockRun(run);
                }else{
                    // read Lock
                    try{
                        lockRead.lock();
                        return run.apply();
                    }finally{
                        lockRead.unlock();
                    }
                }
            }else{
                return lockRun(run);
            }
        }
    }

    @Override
    protected Object lockRun(Func0 run) {
        try{
            lockWrite.lock();
            return run.apply();
        }finally{
            lockWrite.unlock();
        }
    }
}
