/*
 * The MIT License
 *
 * Copyright 2016 Kamnev Georgiy <nt.gocha@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.collection.set;

import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func0;
import xyz.cofe.collection.LockMethod;

/**
 * event set с поддержкой блокировки
 * @author nt.gocha@gmail.com
 */
public class LockEventSet<E> extends BasicEventSet<E>
{
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(LockEventSet.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(LockEventSet.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(LockEventSet.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(LockEventSet.class.getName(), method, result);
    }
    //</editor-fold>
    
    protected final Lock lockWrite;
    protected final Lock lockRead;

    public LockEventSet() {
        super();
        lockWrite = new ReentrantLock();
        lockRead = lockWrite;
    }

    public LockEventSet(Lock lock) {
        super();
        this.lockWrite = lock==null ? new ReentrantLock()  : lock;
        this.lockRead = lockWrite;
    }

    public LockEventSet(Set<E> set) {
        super(set);
        lockWrite = new ReentrantLock();
        lockRead = lockWrite;
    }

    public LockEventSet(Set<E> set, Lock lock) {
        super(set);
        this.lockWrite = lock==null ? new ReentrantLock()  : lock;
        this.lockRead = lockWrite;
    }

    public LockEventSet(Set<E> set, Lock lockRead, Lock lockWrite) {
        super(set);
        this.lockWrite = lockWrite==null ? new ReentrantLock()  : lockWrite;
        this.lockRead = lockRead==null ? this.lockWrite : lockRead;
    }

    @Override
    protected Object lockRun(Func0 run, LockMethod method) {
        if( run==null )throw new IllegalArgumentException( "run==null" );

        if( lockWrite==lockRead ){
            return lockRun(run);
        }else{
            if( method!=null ){
                boolean isRead = method.getWrite()==null || !method.getWrite();
                boolean isWrite = !isRead;
                if( isWrite ){
                    // write Lock
                    return lockRun(run);
                }else{
                    // read Lock
                    try{
                        lockRead.lock();
                        return run.apply();
                    }finally{
                        lockRead.unlock();
                    }
                }
            }else{
                return lockRun(run);
            }
        }
    }

    @Override
    protected Object lockRun(Func0 run) {
        if( run==null )throw new IllegalArgumentException( "run==null" );
        
        try{
            lockWrite.lock();
            return run.apply();
        }finally{
            lockWrite.unlock();
        }
    }
}
