/*
 * The MIT License
 *
 * Copyright 2017 Kamnev Georgiy <nt.gocha@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.collection.tree;

/**
 * Смещение к узлу относительно данного узла. <br>
 * Все узлы древа нумеруются в порядке обхода в глубину. <br>
 * Пример 1(а):
 * <ul>
 * <li>А (0)
 * <ul>
 * <li>B (1)
 * <ul>
 * <li>D (2)</li>
 * <li>E (3)</li>
 * </ul>
 * </li>
 * <li>C (4)
 * <ul>
 * <li>F (5)
 * <ul>
 * <li>H (6)</li>
 * <li>I (7)</li>
 * <li>J (8)</li>
 * </ul>
 * </li>
 * <li>G (9)</li>
 * </ul>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * Пример 1(б):
 * <table border='1' summary="Пример дерева">
 * 
 * <tr>
 * <td align='center' colspan='6'>A (0)</td>
 * </tr>
 * 
 * <tr>
 * <td align='center' colspan='2'>B (1)</td>
 * <td align='center' colspan='4'>C (4)</td>
 * </tr>
 * 
 * <tr>
 * <td align='center' colspan='1'>D (2)</td>
 * <td align='center' colspan='1'>E (3)</td>
 * <td align='center' colspan='3'>F (5)</td>
 * <td align='center' colspan='1'>G (9)</td>
 * </tr>
 * 
 * <tr>
 * <td align='center' colspan='2'> </td>
 * <td align='center' colspan='1'>H (6)</td>
 * <td align='center' colspan='1'>I (7)</td>
 * <td align='center' colspan='1'>J (8)</td>
 * <td align='center' colspan='1'> </td>
 * </tr>
 * </table>
 * <br>
 * <b>deepOffset()</b><br>
 * Смещение +3 относительно узла A вернет узел E, а для узла E вернет узел H<br>
 * <code style='font-size:100%'>a.deepOffset(3) == e</code> <br>
 * <code style='font-size:100%'>e.deepOffset(3) == h</code> <br><br>
 * 
 * Смещение -5 относительно узла I вернет узел D: <br>
 * <code style='font-size:100%'>i.deepOffset(-5) == d</code> <br><br>
 * 
 * <b>getRootOffset()</b><br>
 * getRootOffset - возвращает смещение относительно корня.<br>
 * Так для узла A значение будет 0, а для узла H - 6:<br>
 * <code style='font-size:100%'>a.getRootOffset() == 0</code> <br>
 * <code style='font-size:100%'>h.getRootOffset() == 6</code> <br>
 * @author nt.gocha@gmail.com
 */
public interface TreeNodeDeepOffset<Node extends TreeNode>
       extends TreeNodeDeepCounter<Node>
{
    /**
     * Возвращает узел относительно текущего в древе, движение согласно порядку обхода узлов вглубь.
     * @param offset смещение,<br>
     * для 0 - вернет текущий. <br>
     * @return Узел в древе относительно указанного или null
     */
    public Node deepOffset( int offset );
    
    /**
     * Возвращает смещение узла относительно корня древа
     * @return смещение (0 для корня).
     */
    public int getRootOffset();
}
