/*
 * The MIT License
 *
 * Copyright 2017 user.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package xyz.cofe.collection.tree;

import java.io.Closeable;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func2;
import xyz.cofe.common.ListenersHelper;
import xyz.cofe.common.Reciver;

/**
 * Дефолт реализация подписчиков
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 */
public class TreeNodeListenersDef {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static final Logger logger = Logger.getLogger(TreeNodeListenersDef.class.getName());
    private static final Level logLevel = logger.getLevel();
    
    private static final boolean isLogSevere = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.SEVERE.intValue();
    
    private static final boolean isLogWarning = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.WARNING.intValue();
    
    private static final boolean isLogInfo = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.INFO.intValue();
    
    private static final boolean isLogFine = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINE.intValue();
    
    private static final boolean isLogFiner = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINER.intValue();
    
    private static final boolean isLogFinest = 
        logLevel==null 
        ? true
        : logLevel.intValue() <= Level.FINEST.intValue();

    private static void logFine(String message,Object ... args){
        logger.log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        logger.log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        logger.log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        logger.log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        logger.log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        logger.log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        logger.log(Level.SEVERE, null, ex);
    }

    private static void logEntering(String method,Object ... params){
        logger.entering(TreeNodeListenersDef.class.getName(), method, params);
    }
    
    private static void logExiting(String method){
        logger.exiting(TreeNodeListenersDef.class.getName(), method);
    }
    
    private static void logExiting(String method, Object result){
        logger.exiting(TreeNodeListenersDef.class.getName(), method, result);
    }
    //</editor-fold>

    public static final WeakHashMap<TreeNode,ListenersHelper> listeners
        = new WeakHashMap<TreeNode,ListenersHelper>();
    
    /**
     * Возвращает помошника подписчиков
     * @param self Ссылка на TreeNodeListeners
     * @return помошник подписчиков
     */
    public static ListenersHelper<TreeNodeListener,TreeNodeEvent> getListenersHelper( final TreeNodeListeners self){
        synchronized( listeners ){
            ListenersHelper<TreeNodeListener,TreeNodeEvent>
                listener = listeners.get(self);
            
            if( listener==null ){
                listener = new ListenersHelper<TreeNodeListener,TreeNodeEvent>( 
                    new Func2<Object, TreeNodeListener, TreeNodeEvent>() {
                        @Override
                        public Object apply(TreeNodeListener arg1, TreeNodeEvent arg2) {
                            if( arg1!=null )arg1.treeNodeEvent(arg2);
                            return null;
                        }
                    }
                );
                
                listeners.put(self, listener);
            }
            
            return listener;
        }
    }
    
    /**
     * Добавление подписчика.
     * @param lsrs Ссылка на TreeNodeListeners
     * @param lsr Подписчик
     * @return Интерфес для отсоединения подписчика
     */
    public static Closeable addTreeNodeListener( TreeNodeListeners lsrs, TreeNodeListener lsr ){
        if( lsr==null )throw new IllegalArgumentException( "lsr==null" );
        if( lsrs==null )throw new IllegalArgumentException( "lsrs==null" );
        synchronized( listeners ){
            return getListenersHelper(lsrs).addListener(lsr);
        }
    }
    
    /**
     * Добавление подписчика.
     * @param lsrs Ссылка на TreeNodeListeners
     * @param lsr Подписчик
     * @param weakLink true - добавить как weak ссылку / false - как hard ссылку
     * @return Интерфес для отсоединения подписчика
     */
    public static Closeable addTreeNodeListener( TreeNodeListeners lsrs, TreeNodeListener lsr, boolean weakLink ){
        if( lsr==null )throw new IllegalArgumentException( "lsr==null" );
        if( lsrs==null )throw new IllegalArgumentException( "lsrs==null" );
        synchronized( listeners ){
            return getListenersHelper(lsrs).addListener(lsr,weakLink);
        }
    }
    
    /**
     * Добавление подписчика на события древа
     * @param <EventType> Тип события
     * @param lsrs Ссылка на TreeNodeListeners (издатель)
     * @param cls тип события
     * @param weakLink true - добавить как weak ссылку / false - как hard ссылку
     * @param consumer приемник события (подписчик)
     * @return Интерфес для отсоединения подписчика
     */
    public static <EventType extends TreeNodeEvent> Closeable onTreeNodeEvent( 
        TreeNodeListeners lsrs, 
        final Class<EventType> cls, 
        boolean weakLink, 
        final Reciver<? super EventType> consumer )
    {        
        if( cls==null )throw new IllegalArgumentException( "cls==null" );
        if( consumer==null )throw new IllegalArgumentException( "consumer==null" );
        return addTreeNodeListener( lsrs,
            new TreeNodeListener(){
            @Override
            public void treeNodeEvent(TreeNodeEvent ev) {
                if( ev != null ){
                    Class c = ev.getClass();
                    if( cls.isAssignableFrom(c) ){
                        consumer.recive((EventType)ev);
                    }
                }
            }
            }, weakLink);
    }
    
    /**
     * Добавление подписчика на события древа
     * @param <EventType> Тип события
     * @param lsrs Ссылка на TreeNodeListeners (издатель)
     * @param cls тип события
     * @param consumer приемник события (подписчик)
     * @return Интерфес для отсоединения подписчика
     */
    public static <EventType extends TreeNodeEvent> Closeable onTreeNodeEvent( 
        TreeNodeListeners lsrs, 
        Class<EventType> cls, 
        Reciver<? super EventType> consumer 
    ){
        if( cls==null )throw new IllegalArgumentException( "cls==null" );
        if( consumer==null )throw new IllegalArgumentException( "consumer==null" );
        return onTreeNodeEvent(lsrs, cls, false, consumer);
    }
    
    /**
     * Рассылка уведомления подписчикам
     * @param lsrs Ссылка на TreeNodeListeners (издатель)
     * @param ev уведомление
     */
    public static void fireTreeNodeEvent( TreeNodeListeners lsrs, TreeNodeEvent ev ){
        if( ev==null )throw new IllegalArgumentException( "ev==null" );
        synchronized( listeners ){
            getListenersHelper( lsrs ).fireEvent(ev);
        }
    }

    /**
     * роверка наличия подписчика в списке обработки
     * @param lsrs Ссылка на TreeNodeListeners (издатель)
     * @param lsr подписчик
     * @return есть в списке обработки
     */
    public static boolean hasTreeNodeListener( TreeNodeListeners lsrs, TreeNodeListener lsr ){
        synchronized( listeners ){
            return getListenersHelper( lsrs ).hasListener( lsr );
        }
    }

    /**
     * Удаление подписчика из списка обработки
     * @param lsrs Ссылка на TreeNodeListeners (издатель)
     * @param lsr подписчик
     */
    public static void removeTreeNodeListener( TreeNodeListeners lsrs, TreeNodeListener lsr ){
        synchronized( listeners ){
            getListenersHelper( lsrs ).removeListener( lsr );
        }
    }
    
    /**
     * Получение списка подписчиков
     * @param lsrs Ссылка на TreeNodeListeners (издатель)
     * @return подписчики
     */
    public static Set<TreeNodeListener> getTreeNodeListeners( TreeNodeListeners lsrs ){
        synchronized( listeners ){
            return getListenersHelper( lsrs ).getListeners();
        }
    }
}
