/*
 * The MIT License
 *
 * Copyright 2015 Kamnev Georgiy (nt.gocha@gmail.com).
 *
 * Данная лицензия разрешает, безвозмездно, лицам, получившим копию данного программного 
 * обеспечения и сопутствующей документации (в дальнейшем именуемыми "Программное Обеспечение"), 
 * использовать Программное Обеспечение без ограничений, включая неограниченное право на 
 * использование, копирование, изменение, объединение, публикацию, распространение, сублицензирование 
 * и/или продажу копий Программного Обеспечения, также как и лицам, которым предоставляется 
 * данное Программное Обеспечение, при соблюдении следующих условий:
 *
 * Вышеупомянутый копирайт и данные условия должны быть включены во все копии 
 * или значимые части данного Программного Обеспечения.
 *
 * ДАННОЕ ПРОГРАММНОЕ ОБЕСПЕЧЕНИЕ ПРЕДОСТАВЛЯЕТСЯ «КАК ЕСТЬ», БЕЗ ЛЮБОГО ВИДА ГАРАНТИЙ, 
 * ЯВНО ВЫРАЖЕННЫХ ИЛИ ПОДРАЗУМЕВАЕМЫХ, ВКЛЮЧАЯ, НО НЕ ОГРАНИЧИВАЯСЬ ГАРАНТИЯМИ ТОВАРНОЙ ПРИГОДНОСТИ, 
 * СООТВЕТСТВИЯ ПО ЕГО КОНКРЕТНОМУ НАЗНАЧЕНИЮ И НЕНАРУШЕНИЯ ПРАВ. НИ В КАКОМ СЛУЧАЕ АВТОРЫ 
 * ИЛИ ПРАВООБЛАДАТЕЛИ НЕ НЕСУТ ОТВЕТСТВЕННОСТИ ПО ИСКАМ О ВОЗМЕЩЕНИИ УЩЕРБА, УБЫТКОВ 
 * ИЛИ ДРУГИХ ТРЕБОВАНИЙ ПО ДЕЙСТВУЮЩИМ КОНТРАКТАМ, ДЕЛИКТАМ ИЛИ ИНОМУ, ВОЗНИКШИМ ИЗ, ИМЕЮЩИМ 
 * ПРИЧИНОЙ ИЛИ СВЯЗАННЫМ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ ИЛИ ИСПОЛЬЗОВАНИЕМ ПРОГРАММНОГО ОБЕСПЕЧЕНИЯ 
 * ИЛИ ИНЫМИ ДЕЙСТВИЯМИ С ПРОГРАММНЫМ ОБЕСПЕЧЕНИЕМ.
 */

package xyz.cofe.common;


import java.io.Closeable;
import java.io.IOException;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
//import java.util.function.BiConsumer;
import java.util.logging.Level;
import java.util.logging.Logger;
import xyz.cofe.collection.Func2;

// TODO требует тестирования
/**
 * Помошник в реализации Listener. <br>
 * Обеспечивает синхронность при вызове методов: 
 * <ul>
 * <li>getListeners</li>
 * <li>addListener</li>
 * <li>removeListener</li>
 * <li>fireEvent</li>
 * </ul>
 * Умеет хранить как hard так и soft ссылки на listener-ы
 * @author Kamnev Georgiy (nt.gocha@gmail.com)
 * @param <ListenerType> Тип издателья
 * @param <EventType> Тип подписчика
 */
public class ListenersHelper<ListenerType,EventType> {
    //<editor-fold defaultstate="collapsed" desc="log Функции">
    private static void logFine(String message,Object ... args){
        Logger.getLogger(ListenersHelper.class.getName()).log(Level.FINE, message, args);
    }
    
    private static void logFiner(String message,Object ... args){
        Logger.getLogger(ListenersHelper.class.getName()).log(Level.FINER, message, args);
    }
    
    private static void logFinest(String message,Object ... args){
        Logger.getLogger(ListenersHelper.class.getName()).log(Level.FINEST, message, args);
    }
    
    private static void logInfo(String message,Object ... args){
        Logger.getLogger(ListenersHelper.class.getName()).log(Level.INFO, message, args);
    }

    private static void logWarning(String message,Object ... args){
        Logger.getLogger(ListenersHelper.class.getName()).log(Level.WARNING, message, args);
    }
    
    private static void logSevere(String message,Object ... args){
        Logger.getLogger(ListenersHelper.class.getName()).log(Level.SEVERE, message, args);
    }

    private static void logException(Throwable ex){
        Logger.getLogger(ListenersHelper.class.getName()).log(Level.SEVERE, null, ex);
    }
    //</editor-fold>
    
    /**
     * Объект по которому происходи синхронизация
     */
    protected Lock lock = null;
    
    /**
     * Функция вызывающая listener&#x002e;<i>method</i>( <i>event</i> );
     */
    protected Func2<Object,ListenerType,EventType> callListener = null;
    
    /**
     * Конструктор.
     * По умолчанию синхронизация включена, объект по которому происходит синх - this.
     * @param callListFunc Функция вызывающая listener&#x002e;<i>method</i>( <i>event</i> );
     */
    public ListenersHelper(Func2<Object,ListenerType,EventType> callListFunc){
        if( callListFunc==null )throw new IllegalArgumentException( "callListFunc==null" );
        this.lock = null;
        this.callListener = callListFunc;
    }
    
    /**
     * Конструктор.
     * По умолчанию синхронизация включена, объект по которому происходит синх - this.
     * @param callListFunc Функция вызывающая listener&#x002e;<i>method</i>( <i>event</i> );
    public ListenersHelper(BiConsumer<? super ListenerType,? super EventType> callListFunc){
        if( callListFunc==null )throw new IllegalArgumentException( "callListFunc==null" );
        this.lock = null;
        this.callListener = (listener,event) -> { callListFunc.accept(listener, event); return null; };
    }
     */
    
    /**
     * Конструктор.
     * @param callListFunc Функция вызывающая listener&#x002e;<i>method</i>( <i>event</i> );
     * @param sync Синхронно производить вызовы
     */
    public ListenersHelper(Func2<Object,ListenerType,EventType> callListFunc, Lock sync){
        if( callListFunc==null )throw new IllegalArgumentException( "callListFunc==null" );
        this.lock = sync;
        this.callListener = callListFunc;
    }
    
//    /**
//     * Конструктор.
//     * @param callListFunc Функция вызывающая listener&#x002e;<i>method</i>( <i>event</i> );
//     * @param sync Синхронно производить вызовы
//     * @param synObj Объект по которому происходи синхронизация
//     */
//    public ListenersHelper(Func2<Object,ListenerType,EventType> callListFunc, boolean sync, Object synObj){
//        if( callListFunc==null )throw new IllegalArgumentException( "callListFunc==null" );
////        if( synObj==null )throw new IllegalArgumentException( "synObj==null" );
//        if( sync ){
//            this.sync = synObj;
//        }
//        this.callListener = callListFunc;
//    }
    
    public Lock getLock(){ return lock; }

    /**
     * Hard ссылки на подписчиков
     */
    protected HashSet<ListenerType> listeners = new LinkedHashSet<ListenerType>();
    
    /**
     * Soft/Weak ссылки на подписчиков
     */
    protected WeakHashMap<ListenerType,Object> weakListeners = new WeakHashMap<ListenerType,Object>();
    
    /**
     * Создает блокировку ReentrantLock
     * @return блокировка
     */
    protected Lock createLock(){
        return new ReentrantLock();
    }

    /**
     * Проверка наличия подписчика в списке обработки
     * @param listener подписчик
     * @return true - есть в списке обработки
     */
    public boolean hasListener( ListenerType listener ){
        if( listener==null )return false;
        if( lock!=null )
        {
            lock.lock();
            try{
                if( listeners.contains(listener) )return true;
                return weakListeners.containsKey(listener);
            }finally{
                lock.unlock();
            }
        }else{
            if( listeners.contains(listener) )return true;
            return weakListeners.containsKey(listener);
        }
    }
    
    /**
     * Получение списка подписчиков
     * @return подписчики
     */
    public Set<ListenerType> getListeners() { 
        HashSet<ListenerType> res = new LinkedHashSet<ListenerType>();
        if( lock!=null ){
            lock.lock();
            try{
                res.addAll(listeners);
                res.addAll(weakListeners.keySet());
            }finally{
                lock.unlock();
            }
        }else{
            res.addAll(listeners);
            res.addAll(weakListeners.keySet());
        }
        return res; 
    }
    
    /**
     * Добавление подписчика.
     * @param listener Подписчик.
     * @return Интерфес для отсоединения подписчика
     */
    public Closeable addListener(ListenerType listener) {
        if (listener == null) {
            throw new IllegalArgumentException("listener==null");
        }
        if( lock!=null ){
            lock.lock();
            try{
                return addListener0(listener);
            }finally{
                lock.unlock();
            }
        }
        return addListener0(listener);
    }

    private Closeable addListener0(ListenerType listener) {
        listeners.add(listener);
        final ListenerType lstr = listener;
        return new Closeable() {
            private ListenerType l_str = lstr;
            
            @Override
            public void close() throws IOException {
                if (l_str != null) {
                    ListenersHelper.this.removeListener(l_str);
                    l_str = null;
                }
            }
        };
    }
    
    /**
     * Добавление подписчика.
     * @param listener Подписчик.
     * @param weakLink true - добавить как weak ссылку / false - как hard ссылку
     * @return Интерфес для отсоединения подписчика
     */
    public Closeable addListener(ListenerType listener, boolean weakLink) {
        if( listener==null )throw new IllegalArgumentException( "listener==null" );
        if( lock!=null ){
            lock.lock();
            try{
                if( weakLink ){
                    return addListenerWeak0(listener);
                }else{
                    return addListener0(listener);
                }
            }finally{
                lock.unlock();
            }
        }else{
            if( weakLink ){
                return addListenerWeak0(listener);
            }else{
                return addListener0(listener);
            }
        }
    }
    
    private Closeable addListenerWeak0(ListenerType listener) {
        weakListeners.put(listener, true);
        final ListenerType lstr = listener;
        return new Closeable() {
            private ListenerType l_str = lstr;
            @Override
            public void close() throws IOException {
                if (l_str != null) {
                    ListenersHelper.this.removeListener(l_str);
                    l_str = null;
                }
            }
        };
    }
    
    /**
     * Удаление подписчика из списка обработки
     * @param listener подписчик
     */
    public void removeListener(ListenerType listener) {
        if( lock!=null ){
            lock.lock();
            try{
                if (listener != null) {
                    listeners.remove(listener);
                    weakListeners.remove(listener);
                }
            }finally{
                lock.unlock();
            }
        }else{
            if (listener != null) {
                listeners.remove(listener);
                weakListeners.remove(listener);
            }
        }
    }
    
    /**
     * Рассылка уведомления подписчикам
     * @param event уведомление
     */
    public void fireEvent(EventType event) {
        Object[] lstnrs = null;
        if( lock!=null ){
            lock.lock();
            try{
                lstnrs = getListeners().toArray();
            }finally{
                lock.unlock();
            }
        }else{
            lstnrs = getListeners().toArray();
        }
        if (event != null) {
            for (Object lo : lstnrs) {
                if( lo != null ){
                    ListenerType l = (ListenerType)lo;
                    callListener.apply(l,event);
                }
            }
        }
    }
}
