package xyz.gunanyi.currency.aspect;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.annotation.TableName;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import xyz.gunanyi.currency.annotation.OperLog;
import xyz.gunanyi.currency.entity.SysOperationLog;
import xyz.gunanyi.currency.service.ISysOperationLogService;
import xyz.gunanyi.currency.utils.CusAccessObjectUtil;
import xyz.gunanyi.currency.utils.IpAndAddrUtil;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 切面处理类，操作日志异常日志记录处理
 *
 * @author: 曾博旺
 */
@Aspect
@Component
@Slf4j
public class OperLogAspect {
    /**
     * 操作版本号
     *
     * @Value("${version}") 项目启动时从命令行传入，例如：java -jar xxx.war --version=201902
     */

    private String operVer = System.getProperty("java.version");

    @Autowired
    private ISysOperationLogService sysOperationLogService;

    /**
     * 设置操作日志切入点 记录操作日志 在注解的位置切入代码
     */
    @Pointcut("@annotation(xyz.gunanyi.currency.annotation.OperLog)")
    public void excudeService() {
    }


    /**
     * 正常返回通知，拦截用户操作日志，连接点正常执行完成后执行， 如果连接点抛出异常，则不会执行
     *
     * @param joinPoint 切入点 keys返回值
     * @return Object
     */
    @Around(value = "excudeService()")
    public Object saveOperLog(ProceedingJoinPoint joinPoint) {
        Object result = null;
        // 获取RequestAttributes
        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
        // 从获取RequestAttributes中获取HttpServletRequest的信息
        HttpServletRequest request = (HttpServletRequest) requestAttributes
                .resolveReference(RequestAttributes.REFERENCE_REQUEST);

        SysOperationLog operlog = new SysOperationLog();
        try {

            // 从切面织入点处通过反射机制获取织入点处的方法
            MethodSignature signature = (MethodSignature) joinPoint.getSignature();
            // 获取切入点所在的方法
            Method method = signature.getMethod();

            // 请求的参数
            JSONObject rtnObj = new JSONObject();
            Map<String, String> rtnMap = converMap(request.getParameterMap());
            if (rtnMap.size() == 0) {
                rtnObj = getPara(joinPoint);
            } else {
                String json = JSON.toJSONString(rtnMap);
                rtnObj = JSON.parseObject(json);
            }

            // 将参数所在的数组转换成json
            String params = JSON.toJSONString(rtnObj);
            // 获取操作
            OperLog opLog = method.getAnnotation(OperLog.class);
            if (opLog != null) {
                String operModul = opLog.operModul();
                String operType = opLog.operType();
                String operDesc = opLog.operDesc();
                operlog.setOperModul(operModul);
                operlog.setOperType(operType);
                operlog.setOperDesc(operDesc);
                try {
                    operlog.setBrowser(IpAndAddrUtil.getBrowserName(request)+"\t："+IpAndAddrUtil.getBrowserVersion(request));
                    operlog.setIpaddress(IpAndAddrUtil.getIp(request));
                    operlog.setOs(IpAndAddrUtil.getOsName(request));
                } catch (Exception e) {
                    log.error("地址获取失败");
                    e.printStackTrace();
                }
                //若要记录操作前数据
                if (opLog.isPreData()) {
                    Class clz = opLog.clz();
                    String table = opLog.table();
                    String onlyId = opLog.onlyId();
                    String operData = this.addPerData(clz, table, onlyId, rtnObj);
                    operlog.setOperData(operData);
                    operlog.setOperationTable(table);
                }

            }
            // 获取请求的类名
            String className = joinPoint.getTarget().getClass().getName();
            // 获取请求的方法名
            String methodName = method.getName();
            methodName = className + "." + methodName;

            operlog.setOperMethod(methodName);

            // 请求参数
            operlog.setOperRequParam(params);
            // 返回结果
            try {
                Object[] args = joinPoint.getArgs();
                result = joinPoint.proceed(args);
                operlog.setOperRespParam(JSON.toJSONString(result));
            } catch (Throwable throwable) {
                log.error("获取错误返回值");
            }
            // 请求用户ID
//            operlog.setOperUserId(sysuser.getId());
            // 请求用户名称
//            operlog.setOperUserName(sysuser.getUsername());
            // 请求IP
            operlog.setOperIp(CusAccessObjectUtil.getIpAddress(request));
            // 请求URI
            operlog.setOperUri(request.getRequestURI());
            // 操作版本
            operlog.setOperVer(operVer);
            sysOperationLogService.save(operlog);
        } catch (Exception e) {
            e.printStackTrace();
        }
        Object[] args = joinPoint.getArgs();
        return result;
    }

    /**
     * 获取日志添加前数据
     *
     * @param clz    类
     * @param table  表
     * @param onlyId 唯一标识
     * @param parm   请求参数
     * @return
     */
    private String addPerData(Class clz, String table, String onlyId, JSONObject parm) {
        if (StringUtils.isBlank(table)) {
            TableName tableName = (TableName) clz.getAnnotation(TableName.class);
            table = tableName.value();
        }
//        String onlyIdValue = afterJson.getString(onlyId);
        String onlyIdValue = parm.getString(onlyId);
        List<Map<String, Object>> obj = sysOperationLogService.executeSql(table, onlyId, onlyIdValue);

        return JSON.toJSONString(obj);
    }


    /**
     * 转换request 请求参数
     *
     * @param paramMap request获取的参数数组
     * @return Map
     */
    public Map<String, String> converMap(Map<String, String[]> paramMap) {
        Map<String, String> rtnMap = new HashMap<String, String>();
        for (String key : paramMap.keySet()) {
            rtnMap.put(key, paramMap.get(key)[0]);
        }
        return rtnMap;
    }

    /**
     * 从request中是无法取到参数到，这是因为参数使用@RequestBoy 那么前端传递的参数是放在请求体里面的。
     * 获取请求参数
     *
     * @param joinPoint joinPoint
     * @return JSONObject
     */
    public JSONObject getPara(JoinPoint joinPoint) {
        Object[] args = joinPoint.getArgs();
        Object arg = args[0];
        String s = null;
        JSONObject afterJson = null;
        try {
            s = JSONObject.toJSONString(arg);
            afterJson = JSONObject.parseObject(s);
        } catch (Exception e) {
            Map<String,Object> map=new HashMap<>();
            map.put("errorMessage","获取请求参数异常");
            return JSONObject.parseObject(JSON.toJSONString(map));
        }
        return afterJson;
//        Signature signature = joinPoint.getSignature();
//        MethodSignature methodSignature = (MethodSignature) signature;
//        String[] parameterNames = methodSignature.getParameterNames();
//        Object[] parameterValues = joinPoint.getArgs();
//        Map<String, String> paramMap = new HashMap<>();
//
//        for (int i = 0; i < parameterValues.length; i++) {
//            try {
//                String s = JSON.toJSONString(parameterValues[i]);
//                paramMap.put(parameterNames[i], s);
//            } catch (Exception e) {
//                continue;
//            }
//        }
//        return paramMap;
    }

    private String getRequestPayload(HttpServletRequest req) {
        StringBuilder sb = new StringBuilder();
        BufferedReader bufferedReader = null;
        try {
            ServletInputStream inputStream = req.getInputStream();
            bufferedReader = new BufferedReader(new InputStreamReader(inputStream, "UTF-8"));
        } catch (IOException e) {
            e.printStackTrace();
        }

//        try(BufferedReader reader = req.getReader()) {
        try (BufferedReader reader = bufferedReader) {
            char[] buff = new char[1024];
            int len;
            while ((len = reader.read(buff)) != -1) {
                sb.append(buff, 0, len);
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return sb.toString();
    }

    /**
     * 转换异常信息为字符串
     *
     * @param exceptionName    异常名称
     * @param exceptionMessage 异常信息
     * @param elements         堆栈信息
     * @return String
     */
    public String stackTraceToString(String exceptionName, String exceptionMessage, StackTraceElement[] elements) {
        StringBuffer strbuff = new StringBuffer();
        for (StackTraceElement stet : elements) {
            strbuff.append(stet + "\n");
        }
        String message = exceptionName + ":" + exceptionMessage + "\n\t" + strbuff.toString();
        return message;
    }


    /**
     * 异常返回通知，用于拦截异常日志信息 连接点抛出异常后执行
     *
     * @param joinPoint 切入点
     * @param e         异常信息
     */
//    @AfterThrowing(pointcut = "operExceptionLogPoinCut()", throwing = "e")
//    public void saveExceptionLog(JoinPoint joinPoint, Throwable e) {
//        // 获取RequestAttributes
//        RequestAttributes requestAttributes = RequestContextHolder.getRequestAttributes();
//        // 从获取RequestAttributes中获取HttpServletRequest的信息
//        HttpServletRequest request = (HttpServletRequest) requestAttributes
//                .resolveReference(RequestAttributes.REFERENCE_REQUEST);
//
//        ExceptionLog excepLog = new ExceptionLog();
//        try {
//            // 从切面织入点处通过反射机制获取织入点处的方法
//            MethodSignature signature = (MethodSignature) joinPoint.getSignature();
//            // 获取切入点所在的方法
//            Method method = signature.getMethod();
//            excepLog.setExcId(UuidUtil.get32UUID());
//            // 获取请求的类名
//            String className = joinPoint.getTarget().getClass().getName();
//            // 获取请求的方法名
//            String methodName = method.getName();
//            methodName = className + "." + methodName;
//            // 请求的参数
//            Map<String, String> rtnMap = converMap(request.getParameterMap());
//            // 将参数所在的数组转换成json
//            String params = JSON.toJSONString(rtnMap);
//            excepLog.setExcRequParam(params); // 请求参数
//            excepLog.setOperMethod(methodName); // 请求方法名
//            excepLog.setExcName(e.getClass().getName()); // 异常名称
//            excepLog.setExcMessage(stackTraceToString(e.getClass().getName(), e.getMessage(), e.getStackTrace())); // 异常信息
//            excepLog.setOperUserId(UserShiroUtil.getCurrentUserLoginName()); // 操作员ID
//            excepLog.setOperUserName(UserShiroUtil.getCurrentUserName()); // 操作员名称
//            excepLog.setOperUri(request.getRequestURI()); // 操作URI
//            excepLog.setOperIp(IPUtil.getRemortIP(request)); // 操作员IP
//            excepLog.setOperVer(operVer); // 操作版本号
//            excepLog.setOperCreateTime(new Date()); // 发生异常时间
//
//            exceptionLogService.insert(excepLog);
//
//        } catch (Exception e2) {
//            e2.printStackTrace();
//        }
//
//    }
}
