// SPDX-License-Identifier: Apache-2.0

package chiseltest.simulator

import chiseltest.simulator.ipc.{IPCSimulatorContext, VpiVerilogHarnessGenerator}
import firrtl2.annotations.NoTargetAnnotation
import firrtl2.{AnnotationSeq, CircuitState}

import java.io.IOException

case object VcsBackendAnnotation extends SimulatorAnnotation {
  override def getSimulator: Simulator = VcsSimulator
}

/** VCS specific options */
trait VcsOption extends NoTargetAnnotation

/** adds flags to the invocation of VCS */
case class VcsFlags(flags: Seq[String]) extends VcsOption

/** adds flags to the C++ compiler in the Makefile generated by Vcs */
case class VcsCFlags(flags: Seq[String]) extends VcsOption

/** adds flags to the simulation binary created by VCS */
case class VcsSimFlags(flags: Seq[String]) extends VcsOption

private object VcsSimulator extends Simulator {
  override def name: String = "vcs"

  /** is this simulator installed on the local machine? */
  override def isAvailable: Boolean = {
    val binaryFound = os.proc("which", "vcs").call().exitCode == 0
    binaryFound && majorVersion >= 2019
  }

  private lazy val version: (Int, Int) = {
    try {
      val VcsVersionRegex = """vcs script version : P-([0-9]+)\.(\d+)""".r
      val text = os.proc("vcs", "-ID").call(check = false, stderr = os.Pipe).out.trim()

      VcsVersionRegex.findFirstMatchIn(text).map { m =>
        (m.group(1).toInt, m.group(2).toInt)
      } match {
        case Some(tuple) => tuple
        case None =>
          throw new SimulatorNotFoundException(s"""Could not parse vcs version in string "$text" """)
      }
    } catch {
      case _: IOException =>
        throw new SimulatorNotFoundException(
          s"""Unable to determine VCS version by running command "vcs -ID" is it installed?"""
        )
    }
  }

  /** search the local computer for an installation of this simulator and print versions */
  def findVersions(): Unit = {
    if (isAvailable) {
      val (maj, min) = version
      println(s"Found Vcs $maj.$min")
    }
  }

  private def majorVersion: Int = version._1

  private def minorVersion: Int = version._2

  override def waveformFormats = Seq(WriteVcdAnnotation, WriteVpdAnnotation, WriteFsdbAnnotation)

  /** start a new simulation
    *
    * @param state
    *   LoFirrtl circuit + annotations
    */
  override def createContext(state: CircuitState): SimulatorContext = {
    // we will create the simulation in the target directory
    val targetDir = Compiler.requireTargetDir(state.annotations)
    val toplevel = TopmoduleInfo(state.circuit)

    // Create the VPI files that vcs needs + a custom harness
    val waveformExt = Simulator.getWavformFormat(state.annotations)
    val moduleNames = GetModuleNames(state.circuit)
    val verilogHarness = generateHarness(
      targetDir,
      toplevel,
      moduleNames,
      waveformExt == WriteVpdAnnotation.format,
      waveformExt == WriteFsdbAnnotation.format
    )

    // compile low firrtl to System Verilog for verilator to use
    Compiler.lowFirrtlToSystemVerilog(state, Seq())

    // turn SystemVerilog into simulation binary
    val userSimFlags = state.annotations.collectFirst { case VcsSimFlags(f) => f }.getOrElse(Seq.empty)
    val simCmd = compileSimulation(toplevel.name, targetDir, verilogHarness, state.annotations) ++
      userSimFlags ++
      waveformFlags(targetDir, toplevel.name, state.annotations)

    // show verbose debug messages
    val verbose = state.annotations.contains(SimulatorDebugAnnotation)

    // the binary we created communicates using our standard IPC interface
    new IPCSimulatorContext(simCmd, toplevel, VcsSimulator, verbose)
  }

  private def waveformFlags(targetDir: os.Path, topName: String, annos: AnnotationSeq): Seq[String] = {
    val ext = Simulator.getWavformFormat(annos)
    val dumpfile = targetDir.relativeTo(os.pwd) / s"$topName.$ext"
    if (ext.isEmpty) {
      Seq.empty
    } else if (ext == "vpd") {
      Seq(s"+vcdplusfile=${dumpfile.toString()}")
    } else if (ext == "fsdb") {
      Seq(s"+fsdbfile=${dumpfile.toString()}")
    } else {
      Seq(s"+dumpfile=${dumpfile.toString()}")
    }
  }

  /** executes VCS in order to generate a simulation binary */
  private def compileSimulation(
    topName:        String,
    targetDir:      os.Path,
    verilogHarness: String,
    annos:          AnnotationSeq
  ): Seq[String] = {
    val relTargetDir = targetDir.relativeTo(os.pwd)
    val flags = generateFlags(topName, targetDir, annos)
    val cmd = List("vcs") ++ flags ++ List("-o", topName) ++
      BlackBox.fFileFlags(targetDir) ++ List(s"$topName.sv", verilogHarness, "vpi.cpp")
    val ret = os.proc(cmd).call(cwd = targetDir)

    assert(ret.exitCode == 0, s"vcs command failed on circuit ${topName} in work dir $targetDir")
    val simBinary = relTargetDir / topName
    assert(os.exists(os.pwd / simBinary), s"Failed to generate simulation binary: $simBinary")
    Seq("./" + simBinary.toString())
  }

  private def generateFlags(topName: String, targetDir: os.Path, annos: AnnotationSeq): Seq[String] = {
    // generate C flags
    val userCFlags = annos.collectFirst { case VcsCFlags(f) => f }.getOrElse(Seq.empty)
    val cFlags = DefaultCFlags(targetDir) ++ userCFlags

    // generate FSDB(Verdi) flags
    val verdiFlags = annos.collectFirst { case WriteFsdbAnnotation =>
      // Check whether VERDI_HOME is defined in system environment
      sys.env.get("VERDI_HOME") match {
        case Some(verdiHome) =>
          Seq(
            "-kdb",
            "-P",
            s"$verdiHome/share/PLI/VCS/LINUX64/novas.tab",
            s"$verdiHome/share/PLI/VCS/LINUX64/pli.a"
          )
        case None =>
          require(requirement = false, s"FSDB waveform dump depends on VERDI_HOME, please set the system environment");
          Seq.empty
      }
    }.getOrElse(Seq.empty)

    // combine all flags
    val userFlags = annos.collectFirst { case VcsFlags(f) => f }.getOrElse(Seq.empty)
    val flags = DefaultFlags(topName, cFlags, verdiFlags) ++ userFlags
    flags
  }

  private def DefaultCFlags(targetDir: os.Path) = List(
    "-I$VCS_HOME/include",
    s"-I$targetDir", // TODO: is this actually necessary?
    "-fPIC",
    "-std=c++11"
  )

  private def DefaultFlags(topName: String, cFlags: Seq[String], verdiFlags: Seq[String]) = List(
    "-full64", /* Default using 64-bit VCS*/
    "-quiet", /* Quite mode*/
    "-sverilog", /* Harness is written in SystemVerilog*/
    "-timescale=1ns/1ps", /* Clock Period*/
    /* pp: registers and variables, callbacks, driver, and assertion debug capability*/
    /* dmptf: debug ports and internal nodes/memories of tasks/functions*/
    "-debug_acc+pp+dmptf",
    /* cell: debug both real cell modules and the ports of real cell modules*/
    /* encrypt: debug fully-encrypted instances*/
    "-debug_region+cell+encrypt",
    s"-Mdir=$topName.csrc",
    "+v2k", /* verilog 2000 standard*/
    "+vpi",
    "+vcs+lic+wait", /* wait for VCS license if there isn't one*/
    "+vcs+initreg+random", /* Randomize the register*/
    "+define+CLOCK_PERIOD=1", /* Define the clock period*/
    "-P", /* Pass vpi.tab*/
    "vpi.tab", /* like above*/
    "-cpp",
    "g++",
    "-O2",
    "-LDFLAGS",
    "-lstdc++",
    "-CFLAGS",
    cFlags.mkString(" ") // Don't wrap in double quotes
  ) ++ verdiFlags

  private def generateHarness(
    targetDir:   os.Path,
    toplevel:    TopmoduleInfo,
    moduleNames: Seq[String],
    useVpdDump:  Boolean = false,
    useFsdbDump: Boolean = false
  ): String = {
    val topName = toplevel.name

    // copy the VPI files + generate a custom verilog harness
    CopyVpiFiles(targetDir)
    val verilogHarnessFileName = s"${topName}-harness.sv"
    val emittedStuff = VpiVerilogHarnessGenerator.codeGen(toplevel, moduleNames, useVpdDump, useFsdbDump)

    os.write.over(targetDir / verilogHarnessFileName, emittedStuff)
    verilogHarnessFileName
  }
}

/** Copies the files needed for the VPI based simulator interface.
  */
private object CopyVpiFiles {
  def apply(destinationDirPath: os.Path): Unit = {
    // note: vpi_register.cpp is only used by icarus, not by VCS
    val files = Seq("sim_api.h", "vpi.h", "vpi.cpp", "vpi.tab", "vpi_register.cpp")
    val resourcePrefix = "/simulator/"

    files.foreach { name =>
      val dst = destinationDirPath / name
      val src = getClass.getResourceAsStream(resourcePrefix + name)
      os.write.over(target = dst, data = src, createFolders = true)
    }
  }
}
